/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wmq.step;

import static org.apache.commons.lang.StringUtils.join;

import java.util.ArrayList;
import java.util.List;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.hostsession.CapturingCommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.wmq.ci.WmqQueueManager;
import com.xebialabs.deployit.plugin.wmq.utils.RegexUtils;

/**
 * Sets the first available ServerConnectionChannel to the {@link WmqQueueManager}.
 */
@SuppressWarnings("serial")
public class CheckWmqChannelStep implements Step {

	private WmqQueueManager wmqQueueManager;

	public CheckWmqChannelStep(WmqQueueManager wmqQueueManager) {
		this.wmqQueueManager = wmqQueueManager;
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession hostSession = wmqQueueManager.getHostName().getHostSession();
		List<String> list = new ArrayList<String>();
		try {

			CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
			List<String> cmd = new ArrayList<String>();

			if (wmqQueueManager.getHostName().getOperatingSystemFamily() == OperatingSystemFamily.WINDOWS) {
				cmd.add("cmd");
				cmd.add("/c");
			}
			cmd.add("echo");
			cmd.add("dis");
			cmd.add("chl(*)");
			cmd.add("trptype");
			cmd.add("type(svrconn)");
			cmd.add("|");
			cmd.add("runmqsc");
			cmd.add(wmqQueueManager.getQueueManagerName());

			String[] cmdArray = cmd.toArray(new String[cmd.size()]);

			int res = hostSession.execute(capturedOutput, cmdArray);
			if (res != 0 && res != 10) {
				ctx.logError("Non-zero exitcode from " + join(cmdArray, " ") + ": " + res);
				return false;
			}

			for (String eachLine : capturedOutput.getOutputLines()) {
				if (eachLine.contains("CHANNEL")) {
					String[] matches = RegexUtils.regexMatch(eachLine, "\\s+CHANNEL\\((\\S+)\\)[\\s\\S]*");
					if (matches != null && matches.length == 1) {
						list.add(matches[0]);
					}
				}
			}

		} finally {

			hostSession.close();
		}

		if (list.size() > 0) {

			ctx.logOutput("Available Queue Manager Server Connection Channel Names " + list);
			wmqQueueManager.setChannelName(list.get(0));
			ctx.logOutput("The channel \"" + list.get(0) + "\" is set to the QueueManager \"" + wmqQueueManager.getQueueManagerName() + "\"");
			if (list.size() > 1) {
				ctx.logOutput("User can change the channel on the QueueManager after the discovery process is completed to any of the above mentioned channel names");
			}
			return true;
		}

		ctx.logError("Could not determine Channel Name of Queue Manager " + wmqQueueManager.getQueueManagerName());

		return false;
	}

	public String getDescription() {
		if (this.wmqQueueManager.getQueueManagerName() != null) {
			return "Check Wmq Channel Step for \"" + this.wmqQueueManager.getQueueManagerName() + "\"";
		}
		return "Check Wmq Channel Step";
	}
}