/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.runbook;

import java.io.Serializable;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.ConfigurationFiles;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.mapper.artifact.ConfigurationFilesToHostMapper;
import com.xebialabs.deployit.mapper.artifact.GenericFolderToHostMapper;
import com.xebialabs.deployit.mapper.artifact.LibrariesToHostMapper;
import com.xebialabs.deployit.mapper.artifact.SqlFolderToDatabaseMapper;
import com.xebialabs.deployit.mapper.artifact.SqlScriptToDatabaseMapper;
import com.xebialabs.deployit.plugin.apache.httpd.mapper.StaticContentToApacheHttpdServerMapper;
import com.xebialabs.deployit.plugin.wls.ci.WlsArtifactMapping;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsDataSource;
import com.xebialabs.deployit.plugin.wls.ci.WlsDeploymentStrategy;
import com.xebialabs.deployit.plugin.wls.ci.WlsDomain;
import com.xebialabs.deployit.plugin.wls.ci.WlsJmsQueue;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.ci.WlsTarget;
import com.xebialabs.deployit.plugin.wls.mapper.ApacheHttpdWlsPluginConfigurationToApacheHttpdServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbJarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbJarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.JmsMapperUtils;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsDataSourceToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsDataSourceToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsForeignDestinationToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsForeignJmsConnectionFactoryToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsForeignServerToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsConnectionFactoryToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsDistributedQueueToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsQueueToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsJmsServerToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsMaximumThreadsConstraintToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsSharedLibraryJarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsSharedLibraryJarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WlsWorkManagerToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.step.ActivatePendingChangesStep;
import com.xebialabs.deployit.plugin.wls.step.CreateWlsJmsModuleStep;
import com.xebialabs.deployit.plugin.wls.step.DestroyWlsJmsModuleStep;
import com.xebialabs.deployit.plugin.wls.step.RestartServersThatRequireRestartStep;
import com.xebialabs.deployit.plugin.wls.step.StartStopStepFactory;
import com.xebialabs.deployit.plugin.wls.step.WlsDomainCheckReachableStep;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

/**
 * Supports the initial deployment, re- and undeployment of packages to a {@link WlsCluster WLS Cluster}, as well as modifications to the targeted
 * {@link WlsServer Managed Servers} required for the application.
 * <p/>
 * <h4>Conditions</h4>
 * <p/>
 * This runbook will trigger if the change plan contains the
 * <p/>
 * <ul>
 * <li>addition of a new {@link Deployment} CI
 * <li>modification of an existing {@code Deployment} CI
 * <li>deletion of an existing {@code Deployment} CI
 * </ul>
 * <p/>
 * where the <strong>deployment package contains WLS resources</strong> or if components (e.g. {@link Ear EARs} are <strong>targeted at WLS middleware
 * components.</strong>
 * <p/>
 * The runbook will also handle changes to the following properties of {@link WlsServer Managed Servers} <u>to which a resource has been targeted in the current
 * change plan</u>:
 * <p/>
 * <ul>
 * <li>{@link WlsServer#setPort(int) port}
 * <li>{@link WlsServer#setClasspath(String) classpath}
 * <li>{@link WlsServer#setBootClasspath(String) bootClasspath}
 * <li>{@link WlsServer#setInitHeapSize(int) initHeapSize}
 * <li>{@link WlsServer#setMaxHeapSize(int) maxHeapSize}
 * <li>{@link WlsServer#setArguments(String) arguments}
 * </ul>
 * <p/>
 * Modification of <em>other</em> properties of Managed Servers, or of servers which are <em>not</em> targeted by the current deployment, are <strong>not
 * supported</strong>.
 * <h4>Update Semantics</h4>
 * <p/>
 * If the runbook triggers because an existing {@code Deployment} has been modified, the contents of the previous deployment package are compared with the
 * contents of the new package and the following actions are performed:
 * <p/>
 * <ul>
 * <li>if the new deployment package does <u>not</u> contain an element that was present in the previous package, this is <strong>removed</strong> from the
 * environment
 * <li>if the new deployment package contains an element that was <u>not</u> present in the previous package, it is <strong>added</strong>
 * <li>for <em>binary</em> resources ({@link Ear EARs}, {@link ConfigurationFiles} etc.) that are contained in <u>both</u> the new and previous packages, the
 * current resource is <strong>removed</strong> and the new one <strong>added</strong>
 * <li>for <em>configuration</em> resources ({@link WlsJmsQueue JMS Queues}, {@link WlsDataSource JDBC Datasources} etc.) that are contained in <u>both</u> the
 * new and previous packages, the current resource is <strong>modified</strong> if its properties in the new package do not match those in the previous package
 * </ul>
 * <p/>
 * <h4>Actions</h4>
 * <p/>
 * TODO: Step sequence. Note that JMS Queues are not deleted if non-empty.
 */
public class WlsDeploymentRunBook extends SingleTypeHandlingRunBook<Deployment> implements RunBook {

	public WlsDeploymentRunBook() {
		super(Deployment.class);
	}

	@Override
	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		EarToWlsClusterMapper earToClusterMapper = new EarToWlsClusterMapper(change);
		EarToWlsServerMapper earToServerMapper = new EarToWlsServerMapper(change);
		EjbJarToWlsClusterMapper ejbToClusterMapper = new EjbJarToWlsClusterMapper(change);
		EjbJarToWlsServerMapper ejbToServerMapper = new EjbJarToWlsServerMapper(change);
		WarToWlsClusterMapper warToClusterMapper = new WarToWlsClusterMapper(change);
		WarToWlsServerMapper warToServerMapper = new WarToWlsServerMapper(change);
		WlsSharedLibraryJarToWlsServerMapper libJarToServerMapper = new WlsSharedLibraryJarToWlsServerMapper(change);
		WlsSharedLibraryJarToWlsClusterMapper libJarToClusterMapper = new WlsSharedLibraryJarToWlsClusterMapper(change);

		WlsDataSourceToWlsClusterMapper dataSourceToClusterMapper = new WlsDataSourceToWlsClusterMapper(change);
		WlsDataSourceToWlsServerMapper dataSourceToServerMapper = new WlsDataSourceToWlsServerMapper(change);

		Map<WlsServer, WlsCluster> oldServerClusters = getServerClusters(change.getOldRevision());
		Map<WlsServer, WlsCluster> newServerClusters = getServerClusters(change.getNewRevision());
		
		WlsJmsServerToWlsServerMapper jmsServerToServerMapper = new WlsJmsServerToWlsServerMapper(change,
				oldServerClusters, newServerClusters);
		WlsJmsQueueToWlsServerMapper jmsQueueToServerMapper = new WlsJmsQueueToWlsServerMapper(change, 
				oldServerClusters, newServerClusters);
		sortQueueMappings(jmsQueueToServerMapper);
		
		WlsJmsDistributedQueueToWlsClusterMapper jmsDistributedQueueToClusterMapper = new WlsJmsDistributedQueueToWlsClusterMapper(change);

		WlsJmsConnectionFactoryToWlsClusterMapper jmsConnectionFactoryToClusterMapper = new WlsJmsConnectionFactoryToWlsClusterMapper(change);
		WlsForeignServerToWlsClusterMapper jmsForeignServerToClusterMapper = new WlsForeignServerToWlsClusterMapper(change);
		WlsForeignDestinationToWlsClusterMapper jmsForeignDestinationToClusterMapper = new WlsForeignDestinationToWlsClusterMapper(change);
		WlsForeignJmsConnectionFactoryToWlsClusterMapper jmsForeignJmsConnectionFactoryToClusterMapper = new WlsForeignJmsConnectionFactoryToWlsClusterMapper(
				change);

		WlsMaximumThreadsConstraintToWlsClusterMapper maxThreadsConstraintsToClusterMapper = new WlsMaximumThreadsConstraintToWlsClusterMapper(change);
		WlsWorkManagerToWlsClusterMapper workManagerToClusterMapper = new WlsWorkManagerToWlsClusterMapper(change);

		ApacheHttpdWlsPluginConfigurationToApacheHttpdServerMapper pluginConfigMapper = new ApacheHttpdWlsPluginConfigurationToApacheHttpdServerMapper(change);
		StaticContentToApacheHttpdServerMapper staticContentMapper = new StaticContentToApacheHttpdServerMapper(change);
		LibrariesToHostMapper librariesMapper = new LibrariesToHostMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);
		GenericFolderToHostMapper genericFolderToHostMapper = new GenericFolderToHostMapper(change);
		SqlFolderToDatabaseMapper sqlFoldermapper = new SqlFolderToDatabaseMapper(change);
		SqlScriptToDatabaseMapper sqlScriptMapper = new SqlScriptToDatabaseMapper(change);

		Set<WlsCluster> allClusters = new HashSet<WlsCluster>();
		allClusters.addAll(dataSourceToClusterMapper.getAllTargets());
		allClusters.addAll(jmsConnectionFactoryToClusterMapper.getAllTargets());
		allClusters.addAll(jmsForeignServerToClusterMapper.getAllTargets());
		allClusters.addAll(jmsForeignDestinationToClusterMapper.getAllTargets());
		allClusters.addAll(jmsForeignJmsConnectionFactoryToClusterMapper.getAllTargets());
		allClusters.addAll(jmsDistributedQueueToClusterMapper.getAllTargets());
		allClusters.addAll(maxThreadsConstraintsToClusterMapper.getAllTargets());
		allClusters.addAll(workManagerToClusterMapper.getAllTargets());
		allClusters.addAll(earToClusterMapper.getAllTargets());
		allClusters.addAll(warToClusterMapper.getAllTargets());
		allClusters.addAll(ejbToClusterMapper.getAllTargets());
		allClusters.addAll(libJarToClusterMapper.getAllTargets());

		Set<WlsServer> allServers = new HashSet<WlsServer>();
		allServers.addAll(dataSourceToServerMapper.getAllTargets());
		allServers.addAll(jmsServerToServerMapper.getAllTargets());
		allServers.addAll(jmsQueueToServerMapper.getAllTargets());
		allServers.addAll(earToServerMapper.getAllTargets());
		allServers.addAll(warToServerMapper.getAllTargets());
		allServers.addAll(ejbToServerMapper.getAllTargets());
		allServers.addAll(libJarToServerMapper.getAllTargets());

		if (!isWlsDeployment(allClusters, allServers)) {
			return;
		}

		Set<WlsCluster> affectedClusters = new HashSet<WlsCluster>();
		affectedClusters.addAll(dataSourceToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(jmsConnectionFactoryToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(jmsForeignServerToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(jmsForeignDestinationToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(jmsForeignJmsConnectionFactoryToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(jmsDistributedQueueToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(maxThreadsConstraintsToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(workManagerToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(earToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(warToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(ejbToClusterMapper.getAffectedTargets());
		affectedClusters.addAll(libJarToClusterMapper.getAffectedTargets());

		Set<WlsServer> affectedServers = new HashSet<WlsServer>();
		affectedServers.addAll(dataSourceToServerMapper.getAffectedTargets());
		affectedServers.addAll(jmsServerToServerMapper.getAffectedTargets());
		affectedServers.addAll(jmsQueueToServerMapper.getAffectedTargets());
		affectedServers.addAll(earToServerMapper.getAffectedTargets());
		affectedServers.addAll(warToServerMapper.getAffectedTargets());
		affectedServers.addAll(ejbToServerMapper.getAffectedTargets());
		affectedServers.addAll(libJarToServerMapper.getAffectedTargets());

		Set<WlsDomain> affectedDomains = new HashSet<WlsDomain>();
		for (WlsCluster eachCluster : affectedClusters) {
			affectedDomains.add(eachCluster.getDomain());
		}
		for (WlsServer eachServer : affectedServers) {
			affectedDomains.add(eachServer.getDomain());
		}

		Set<WlsServer> oldJmsTargetedServers = new HashSet<WlsServer>();
		oldJmsTargetedServers.addAll(jmsServerToServerMapper.getOldTargets());
		oldJmsTargetedServers.addAll(jmsQueueToServerMapper.getOldTargets());

		Set<WlsServer> newJmsTargetedServers = new HashSet<WlsServer>();
		newJmsTargetedServers.addAll(jmsServerToServerMapper.getNewTargets());
		newJmsTargetedServers.addAll(jmsQueueToServerMapper.getNewTargets());

		Set<WlsCluster> oldJmsTargetedClusters = new HashSet<WlsCluster>();
		oldJmsTargetedClusters.addAll(jmsConnectionFactoryToClusterMapper.getOldTargets());
		oldJmsTargetedClusters.addAll(jmsForeignServerToClusterMapper.getOldTargets());
		oldJmsTargetedClusters.addAll(jmsForeignDestinationToClusterMapper.getOldTargets());
		oldJmsTargetedClusters.addAll(jmsForeignJmsConnectionFactoryToClusterMapper.getOldTargets());
		oldJmsTargetedClusters.addAll(jmsDistributedQueueToClusterMapper.getOldTargets());

		Set<WlsCluster> newJmsTargetedClusters = new HashSet<WlsCluster>();
		newJmsTargetedClusters.addAll(jmsConnectionFactoryToClusterMapper.getNewTargets());
		newJmsTargetedClusters.addAll(jmsForeignServerToClusterMapper.getNewTargets());
		newJmsTargetedClusters.addAll(jmsForeignDestinationToClusterMapper.getNewTargets());
		newJmsTargetedClusters.addAll(jmsForeignJmsConnectionFactoryToClusterMapper.getNewTargets());
		newJmsTargetedClusters.addAll(jmsDistributedQueueToClusterMapper.getNewTargets());

		for (WlsDomain domain : affectedDomains)
			steps.add(new WlsDomainCheckReachableStep(domain));


		for (WlsTarget target : getTargets(change)) {
			steps.addAll(new StartStopStepFactory().getStopStep(target));
		}

		// undeploy apps
		int lastStepCount = steps.size();
		earToClusterMapper.generateUndeploySteps(steps);
		earToServerMapper.generateUndeploySteps(steps);
		warToClusterMapper.generateUndeploySteps(steps);
		warToServerMapper.generateUndeploySteps(steps);
		ejbToClusterMapper.generateUndeploySteps(steps);
		ejbToServerMapper.generateUndeploySteps(steps);
		libJarToClusterMapper.generateUndeploySteps(steps);
		libJarToServerMapper.generateUndeploySteps(steps);

		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// destroy resources
		lastStepCount = steps.size();
		jmsDistributedQueueToClusterMapper.generateDeletionSteps(steps);
		jmsQueueToServerMapper.generateDeletionSteps(steps);
		jmsConnectionFactoryToClusterMapper.generateDeletionSteps(steps);
		jmsServerToServerMapper.generateDeletionSteps(steps);
		jmsForeignDestinationToClusterMapper.generateDeletionSteps(steps);
		jmsForeignJmsConnectionFactoryToClusterMapper.generateDeletionSteps(steps);
		jmsForeignServerToClusterMapper.generateDeletionSteps(steps);
		
		// hackadoodledoo!
		for (WlsServer each : oldJmsTargetedServers) {
			if (!newJmsTargetedServers.contains(each) && each.isRunningVersionGreaterThan9()) {
				if (oldServerClusters.containsKey(each)) {
					oldJmsTargetedClusters.add(oldServerClusters.get(each));
				} else {
					steps.add(new DestroyWlsJmsModuleStep(each.getDomain(), JmsMapperUtils.getJmsModuleName(change, each)));
				}
			}
		}
		
		for (WlsCluster each : oldJmsTargetedClusters) {
			if (!newJmsTargetedClusters.contains(each) && each.isRunningVersionGreaterThan9()) {
				steps.add(new DestroyWlsJmsModuleStep(each.getDomain(), JmsMapperUtils.getJmsModuleName(change, each)));
			}
		}
		dataSourceToServerMapper.generateDeletionSteps(steps);
		dataSourceToClusterMapper.generateDeletionSteps(steps);
		workManagerToClusterMapper.generateDeletionSteps(steps);
		maxThreadsConstraintsToClusterMapper.generateDeletionSteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		configurationFilesToHostMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);
		genericFolderToHostMapper.generateDeletionSteps(steps);

		genericFolderToHostMapper.generateAdditionSteps(steps);
		librariesMapper.generateDeletionSteps(steps);
		librariesMapper.generateAdditionSteps(steps);

		// Modify any servers if these were changed too, but delegate that to the WlsServerRunBook
		if (change.isModification()) {
			Set<WlsServer> oldApplicationTargetServers = new HashSet<WlsServer>();
			oldApplicationTargetServers.addAll(earToServerMapper.getOldTargets());
			oldApplicationTargetServers.addAll(warToServerMapper.getOldTargets());
			oldApplicationTargetServers.addAll(ejbToServerMapper.getOldTargets());

			Set<WlsCluster> oldApplicationTargetClusters = new HashSet<WlsCluster>();
			oldApplicationTargetClusters.addAll(earToClusterMapper.getOldTargets());
			oldApplicationTargetClusters.addAll(warToClusterMapper.getOldTargets());
			oldApplicationTargetClusters.addAll(ejbToClusterMapper.getOldTargets());
			oldApplicationTargetClusters.addAll(libJarToClusterMapper.getOldTargets());

			for (WlsCluster cluster : oldApplicationTargetClusters) {
				oldApplicationTargetServers.addAll(cluster.getServers());
			}

			Set<WlsServer> newApplicationTargetServers = new HashSet<WlsServer>();
			newApplicationTargetServers.addAll(earToServerMapper.getNewTargets());
			newApplicationTargetServers.addAll(warToServerMapper.getNewTargets());
			newApplicationTargetServers.addAll(ejbToServerMapper.getNewTargets());

			Set<WlsCluster> newApplicationTargetClusters = new HashSet<WlsCluster>();
			newApplicationTargetClusters.addAll(earToClusterMapper.getNewTargets());
			newApplicationTargetClusters.addAll(warToClusterMapper.getNewTargets());
			newApplicationTargetClusters.addAll(ejbToClusterMapper.getNewTargets());
			newApplicationTargetClusters.addAll(libJarToClusterMapper.getNewTargets());

			for (WlsCluster cluster : newApplicationTargetClusters) {
				newApplicationTargetServers.addAll(cluster.getServers());
			}

			Set<Change<WlsServer>> serverChangesForDeploymentChange = findServerChangesForDeploymentChange(changePlan, change, oldApplicationTargetServers,
					newApplicationTargetServers);
			if (serverChangesForDeploymentChange != null) {
				WlsServerRunBook wlsServerRunBook = new WlsServerRunBook();
				for (Change<WlsServer> eachServerChange : serverChangesForDeploymentChange) {
					steps.addAll(wlsServerRunBook.getModificationSteps(eachServerChange));
				}
			}
		}

		// modify queues
		lastStepCount = steps.size();
		jmsQueueToServerMapper.generateModificationSteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		sqlFoldermapper.generateAdditionSteps(steps);
		sqlScriptMapper.generateAdditionSteps(steps);

		// create resources
		lastStepCount = steps.size();
		
		// hackadoodledoo!
		for (WlsServer each : newJmsTargetedServers) {
			if (!oldJmsTargetedServers.contains(each) && each.isRunningVersionGreaterThan9()) {
				if (newServerClusters.containsKey(each)) {
					newJmsTargetedClusters.add(newServerClusters.get(each));
				} else {
					steps.add(new CreateWlsJmsModuleStep(each, JmsMapperUtils.getJmsModuleName(change, each)));
				}
			}
		}
		
		for (WlsCluster each : newJmsTargetedClusters) {
			if (!oldJmsTargetedClusters.contains(each) && each.isRunningVersionGreaterThan9()) {
				steps.add(new CreateWlsJmsModuleStep(each, JmsMapperUtils.getJmsModuleName(change, each)));
			}
		}

		jmsServerToServerMapper.generateAdditionSteps(steps);
		jmsConnectionFactoryToClusterMapper.generateAdditionSteps(steps);
		jmsQueueToServerMapper.generateAdditionSteps(steps);
		jmsForeignServerToClusterMapper.generateAdditionSteps(steps);
		jmsForeignJmsConnectionFactoryToClusterMapper.generateAdditionSteps(steps);
		jmsForeignDestinationToClusterMapper.generateAdditionSteps(steps);
		jmsDistributedQueueToClusterMapper.generateAdditionSteps(steps);
		dataSourceToClusterMapper.generateAdditionSteps(steps);
		dataSourceToServerMapper.generateAdditionSteps(steps);
		maxThreadsConstraintsToClusterMapper.generateAdditionSteps(steps);
		workManagerToClusterMapper.generateAdditionSteps(steps);
		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// deploy apps
		lastStepCount = steps.size();
		earToClusterMapper.generateDeploySteps(steps);
		earToServerMapper.generateDeploySteps(steps);
		warToClusterMapper.generateDeploySteps(steps);
		warToServerMapper.generateDeploySteps(steps);
		ejbToClusterMapper.generateDeploySteps(steps);
		ejbToServerMapper.generateDeploySteps(steps);
		libJarToClusterMapper.generateDeploySteps(steps);
		libJarToServerMapper.generateDeploySteps(steps);

		if (steps.size() > lastStepCount) {
			activateChanges(steps, affectedDomains);
		}

		// start apps
		earToClusterMapper.generateStartSteps(steps);
		earToServerMapper.generateStartSteps(steps);
		warToClusterMapper.generateStartSteps(steps);
		warToServerMapper.generateStartSteps(steps);
		ejbToClusterMapper.generateStartSteps(steps);
		ejbToServerMapper.generateStartSteps(steps);
		libJarToClusterMapper.generateStartSteps(steps);
		libJarToServerMapper.generateStartSteps(steps);

		restartServersRequiringRestart(steps, affectedDomains);

		//start Servers
		for (WlsTarget target : getTargets(change)) {
			steps.addAll(new StartStopStepFactory().getStartStep(target));
		}

		staticContentMapper.generateDeletionSteps(steps);
		staticContentMapper.generateAdditionSteps(steps);
		pluginConfigMapper.generateDeletionSteps(steps);
		pluginConfigMapper.generateAdditionSteps(steps);

		//Nothing to do, remove all the WlsDomainCheckReachableStep
		if (steps.size() == affectedDomains.size())
			steps.clear();
	}

	private static boolean isWlsDeployment(Set<WlsCluster> targetClusters, Set<WlsServer> targetServers) {
		return !targetClusters.isEmpty() || !targetServers.isEmpty();
	}

	private static Map<WlsServer, WlsCluster> getServerClusters(Deployment deployment) {
		Map<WlsServer, WlsCluster> serverClusters = Maps.newHashMap();
		if (deployment == null) {
			return serverClusters;
		}
		
		for (WlsCluster cluster : deployment.getTarget().getMembersOfType(WlsCluster.class)) {
			for (WlsServer server : cluster.getServers()) {
				serverClusters.put(server, cluster);
			}
		}
		return serverClusters;
	}

	private static void sortQueueMappings(WlsJmsQueueToWlsServerMapper queueMapper) {
		Collections.sort(queueMapper.getAddedMappings(), new WlqJmsQueueSorter(true));
		Collections.sort(queueMapper.getDeletedMappings(), new WlqJmsQueueSorter(false));
	}

	@SuppressWarnings("rawtypes")
	private static final class WlqJmsQueueSorter implements Comparator<Mapping> {
		private boolean errorQueuesFirst;

		public WlqJmsQueueSorter(boolean errorQueuesFirst) {
			this.errorQueuesFirst = errorQueuesFirst;
		}

		public int compare(Mapping o1, Mapping o2) {
			WlsJmsQueue lhs = (WlsJmsQueue) o1.getSource();
			WlsJmsQueue rhs = (WlsJmsQueue) o2.getSource();
			int res;
			if (lhs.getErrorQueue() == null) {
				if (rhs.getErrorQueue() == null) {
					res = 0;
				} else {
					res = 1;
				}
			} else {
				if (rhs.getErrorQueue() == null) {
					res = -1;
				} else {
					res = 0;
				}
			}
			if (res == 0) {
				res = lhs.getName().compareTo(rhs.getName());
			}
			if (errorQueuesFirst) {
				res = -res;
			}
			return res;
		}
	}

	private void activateChanges(List<Step> steps, Set<WlsDomain> domains) {
		for (WlsDomain eachDomain : domains) {
			if (eachDomain.isRunningVersionGreaterThan9())
				steps.add(new ActivatePendingChangesStep(eachDomain));
		}
	}

	private void restartServersRequiringRestart(List<Step> steps, Set<WlsDomain> domains) {
		for (WlsDomain eachDomain : domains) {
			if (eachDomain.isUseNodeManager() && eachDomain.isRunningVersionGreaterThan9())
				steps.add(new RestartServersThatRequireRestartStep(eachDomain));
		}
	}

	private Set<Change<WlsServer>> findServerChangesForDeploymentChange(ChangePlan changePlan, Change<Deployment> deploymentChange,
	                                                                    Set<WlsServer> oldTargetServers, Set<WlsServer> newTargetServers) {
		Set<Change<WlsServer>> serverChanges = new HashSet<Change<WlsServer>>();

		for (Change<WlsServer> serverChange : getChangesForCiType(changePlan, WlsServer.class)) {
			if (serverChange.isModification()) {
				if (oldTargetServers.contains(serverChange.getOldRevision()) && newTargetServers.contains(serverChange.getNewRevision())) {
					serverChanges.add(serverChange);
				}
			}
		}
		return serverChanges;
	}

	@SuppressWarnings("rawtypes")
	private Set<Change> getChangesForCiType(ChangePlan cp, Class<? extends Serializable> changeTypeClass) {
		Set<Change> changesOfType = new HashSet<Change>();
		for (Change change : cp.getChanges()) {
			if (changeTypeClass.equals(change.getConfigurationItemClass())) {
				changesOfType.add(change);
			}
		}
		return changesOfType;

	}

	private Set<WlsTarget> getTargets(Change<Deployment> change) {
		final Set<WlsTarget> servers = Sets.newTreeSet(new Comparator<WlsTarget>() {
			public int compare(WlsTarget wlsTarget, WlsTarget wlsTarget1) {
				return wlsTarget.getName().compareTo(wlsTarget1.getName());
			}
		});
		getTargets(servers, change.getNewRevision());
		getTargets(servers, change.getOldRevision());
		return servers;
	}

	@SuppressWarnings("rawtypes")
	private void getTargets(Set<WlsTarget> targets, Deployment deployment) {
		if (deployment == null)
			return;

		final Set<Mapping> set = deployment.getMappings();
		for (Mapping m : set) {
			if (m instanceof WlsArtifactMapping) {
				WlsArtifactMapping wlsArtifactMapping = (WlsArtifactMapping) m;
				if (wlsArtifactMapping.getDeploymentStrategy().equals(WlsDeploymentStrategy.STOP_START)) {

					final WlsTarget target = wlsArtifactMapping.getTarget();
					if (target instanceof WlsServer)
						targets.add(target);
					if (target instanceof WlsCluster) {
						WlsCluster c = (WlsCluster) target;
						for (WlsServer s : c.getServers()) {
							targets.add(s);
						}
					}
				}
			}
		}
	}
}