package com.xebialabs.deployit.plugin.wls.runbook;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.ChangeResolution;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsDomain;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.EjbToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsClusterMapper;
import com.xebialabs.deployit.plugin.wls.mapper.WarToWlsServerMapper;
import com.xebialabs.deployit.plugin.wls.step.CreateWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.DestroyWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.ModifyWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.StartWlsServerStep;
import com.xebialabs.deployit.plugin.wls.step.StopWlsServerStep;

/**
 * Supports the creation, modification and deletion of a {@link WlsServer Managed Server}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or deletion of a {@code WebLogicServer} CI.
 * 
 * <h4>Actions</h4>
 * 
 * Addition:
 * 
 * <ol>
 * <li>A default server is created
 * <li>The server is modified
 * </ol>
 * 
 * Modification:
 * 
 * <ol>
 * <li>The server is stopped
 * <li>The server is modified
 * <li>The server is started
 * </ol>
 * 
 * Deletion:
 * 
 * <ol>
 * <li>The server is stopped
 * <li>The server is deleted
 * </ol>
 */
public class WlsServerRunBook implements RunBook {

	@SuppressWarnings("unchecked")
	public Collection<ChangeResolution> resolve(ChangePlan changePlan) {
		Collection<ChangeResolution> changeResolutions = new ArrayList<ChangeResolution>();

		for (Change<?> change : changePlan.getChanges()) {
			if (change.getConfigurationItemClass().equals(WlsServer.class)) {
				if (change.isDeletion()) {
					changeResolutions.add(new ChangeResolution(change, getDeletionSteps((Change<WlsServer>) change)));
				}
				if (change.isAddition()) {
					changeResolutions.add(new ChangeResolution(change, getCreationSteps((Change<WlsServer>) change)));
				}
				if (change.isModification() && !modificationIsHandledByWlsDeploymentRunBook(changePlan, (Change<WlsServer>) change)) {
					changeResolutions.add(new ChangeResolution(change, getModificationSteps((Change<WlsServer>) change)));
				}
			}
		}

		return changeResolutions;
	}

	@SuppressWarnings("unchecked")
	private boolean modificationIsHandledByWlsDeploymentRunBook(ChangePlan changePlan, Change<WlsServer> serverChange) {
		for (Change<Deployment> deploymentChange : getChangesForCiType(changePlan, Deployment.class)) {

			if (deploymentChange.isModification()) {
				EarToWlsServerMapper earToServerMapper = new EarToWlsServerMapper(deploymentChange);
				EjbToWlsServerMapper ejbToServerMapper = new EjbToWlsServerMapper(deploymentChange);
				WarToWlsServerMapper warToServerMapper = new WarToWlsServerMapper(deploymentChange);
				EarToWlsClusterMapper earToWlsClusterMapper = new EarToWlsClusterMapper(deploymentChange);
				WarToWlsClusterMapper warToWlsClusterMapper = new WarToWlsClusterMapper(deploymentChange);
				EjbToWlsClusterMapper ejbToWlsClusterMapper = new EjbToWlsClusterMapper(deploymentChange);

				Set<WlsServer> oldTargetServers = new HashSet<WlsServer>();
				oldTargetServers.addAll(earToServerMapper.getOldTargets());
				oldTargetServers.addAll(warToServerMapper.getOldTargets());
				oldTargetServers.addAll(ejbToServerMapper.getOldTargets());

				Set<WlsCluster> oldTargetClusters = new HashSet<WlsCluster>();
				oldTargetClusters.addAll(earToWlsClusterMapper.getOldTargets());
				oldTargetClusters.addAll(warToWlsClusterMapper.getOldTargets());
				oldTargetClusters.addAll(ejbToWlsClusterMapper.getOldTargets());
				for (WlsCluster cluster : oldTargetClusters) {
					oldTargetServers.addAll(cluster.getServers());
				}

				Set<WlsServer> newTargetServers = new HashSet<WlsServer>();
				newTargetServers.addAll(earToServerMapper.getNewTargets());
				newTargetServers.addAll(warToServerMapper.getNewTargets());
				newTargetServers.addAll(ejbToServerMapper.getNewTargets());

				Set<WlsCluster> newTargetClusters = new HashSet<WlsCluster>();
				newTargetClusters.addAll(earToWlsClusterMapper.getNewTargets());
				newTargetClusters.addAll(warToWlsClusterMapper.getNewTargets());
				newTargetClusters.addAll(ejbToWlsClusterMapper.getNewTargets());
				for (WlsCluster cluster : newTargetClusters) {
					newTargetServers.addAll(cluster.getServers());
				}

				if (oldTargetServers.contains(serverChange.getOldRevision()) && newTargetServers.contains(serverChange.getNewRevision())) {
					return true;
				}
			}
		}
		return false;
	}

	private Set<Change> getChangesForCiType(ChangePlan cp, Class<? extends Serializable> changeTypeClass) {
		Set<Change> changesOfType = new HashSet<Change>();
		for (Change change : cp.getChanges()) {
			if (changeTypeClass.equals(change.getConfigurationItemClass())) {
				changesOfType.add(change);
			}
		}
		return changesOfType;

	}

	public List<Step> getCreationSteps(Change<WlsServer> change) {
		WlsServer server = change.getNewRevision();

		List<Step> allSteps = new ArrayList<Step>();
		allSteps.add(new CreateWlsServerStep(server));
		allSteps.add(new ModifyWlsServerStep(server));
		allSteps.add(new StartWlsServerStep(server));

		return allSteps;
	}

	public List<Step> getDeletionSteps(Change<WlsServer> change) {
		WlsServer server = change.getOldRevision();
		WlsDomain domain = server.getDomain();

		List<Step> allSteps = new ArrayList<Step>();
		allSteps.add(new StopWlsServerStep(server));
		allSteps.add(new DestroyWlsServerStep(server));

		return allSteps;
	}

	public List<Step> getModificationSteps(Change<WlsServer> change) {
		WlsServer server = change.getNewRevision();
		WlsDomain domain = server.getDomain();

		List<Step> allSteps = new ArrayList<Step>();
		allSteps.add(new StopWlsServerStep(server));
		allSteps.add(new ModifyWlsServerStep(server));
		allSteps.add(new StartWlsServerStep(server));

		return allSteps;
	}

}
