/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.runbook;

import com.xebialabs.deployit.*;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.StaticContent;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.ci.artifact.mapping.DeployableArtifactMapping;
import com.xebialabs.deployit.mapper.Mapper;
import com.xebialabs.deployit.mapper.artifact.*;
import com.xebialabs.deployit.plugin.apache.httpd.ci.ApacheHttpdServer;
import com.xebialabs.deployit.plugin.apache.httpd.mapper.StaticContentToApacheHttpdServerMapper;
import com.xebialabs.deployit.plugin.apache.httpd.step.CreateApacheHttpdVirtualHostStep;
import com.xebialabs.deployit.plugin.apache.httpd.step.DestroyApacheHttpdVirtualHostStep;
import com.xebialabs.deployit.plugin.apache.httpd.step.RestartApacheHttpdServerStep;
import com.xebialabs.deployit.plugin.was.ci.*;
import com.xebialabs.deployit.plugin.was.mapper.*;
import com.xebialabs.deployit.plugin.was.step.SynchronizeWasNodeStep;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

import java.util.*;

/**
 * Supports the deployment, re-deployment and undeployment of a {@link com.xebialabs.deployit.ci.DeploymentPackage Application Package} to a
 * {@link com.xebialabs.deployit.plugin.was.ci.WasCluster WAS Cluster}.
 * <p/>
 * <h4>Conditions</h4>
 * <p/>
 * Will trigger if the change plan contains the addition, modification or deletion of a {@link Deployment Deployment} CI to an
 * {@link com.xebialabs.deployit.ci.Environment Environment} CI which contains WAS middleware CI's.
 * <p/>
 * <h4>Actions</h4>
 * <p/>
 * Addition:
 * <p/>
 * <ol>
 * <li>Deploy {@link WasResource WebSphere resources} to {@link WasTarget target (implemented by WasCluster, WasManagedServer and WasUnmanagedServer)}</li>
 * <li>Deploy {@link WasSharedLibrary} To {@link WasScope} (implemented by {@link WasNode} and {@link WasTarget})</li>
 * <li>Deploy {@link Ear ear} to {@link WasTarget target (implemented by WasCluster, WasManagedServer and WasUnmanagedServer) (Undeploy ear, destroy vhost
 * definition, update webserver plugin config) </li>
 * <li>Deploy {@link War war} to {@link WasTarget target (implemented by WasCluster, WasManagedServer and WasUnmanagedServer) (Undeploy war, destroy vhost
 * definition, update webserver plugin config) </li>
 * <li>Create Apache Virtual Host definition on {@link WasManagedApacheHttpdServer an apache webserver}</li>
 * <li>Deploy {@link StaticContent static content} from {@link WasManagedApacheHttpdServer an apache webserver}</li>
 * <li>Restart {@link WasManagedApacheHttpdServer an apache webserver}</li>
 * </ol>
 * <p/>
 * All Modifications except that for {@link WasSharedLibrary} are handled as Deletion and Addition
 * <p/>
 * <p/>
 * Deletion:
 * <p/>
 * <ol>
 * <li>Undeploy {@link StaticContent static content} from {@link WasManagedApacheHttpdServer an apache webserver}</li>
 * <li>Undeploy {@link Ear ear} from {@link WasTarget target (implemented by WasCluster, WasManagedServer and WasUnmanagedServer) (Undeploy ear, destroy vhost
 * definition, update webserver plugin config)</li>
 * <li>Undeploy {@link War war} from {@link WasTarget target (implemented by WasCluster, WasManagedServer and WasUnmanagedServer) (Undeploy war, destroy vhost
 * definition, update webserver plugin config)</li>
 * <li>Undeploy {@link WasSharedLibrary} from {@link WasScope}(implemented by {@link WasNode} and {@link WasTarget})</li>
 * <li>Undeploy {@link WasResource WebSphere resources} from {@link WasCluster cluster}</li>
 * <li>Destroy Apache Virtual Host definition on {@link WasManagedApacheHttpdServer an apache webserver}</li>
 * <li>Restart {@link WasManagedApacheHttpdServer an apache webserver}</li>
 * </ol>
 * <p/>
 * <h4>Using Httpd servers in Deployment</h4>
 * A list of {@link WasManagedApacheHttpdServer} can be selected from {@link WasEarMapping} and {@link WasWarMapping} if the Ear/War needs to be exposed through WebServers
 * <p/>
 * <h4>Using Was Shared Libraries in Deployments</h4>
 * A list of {@link WasSharedLibrary} can be selected from {@link WasEarMapping}, {@link WasWarMapping} or {@link WasEjbJarMapping} if the Ear/War/EjbJar needs to reference
 * a set of WasSharedLibraries
 */
public class WasDeploymentRunBook extends SingleTypeHandlingRunBook<Deployment> implements RunBook {

	public WasDeploymentRunBook() {
		super(Deployment.class);
	}

	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		Set<WasScope> affectedWasTargets = new HashSet<WasScope>();
		EarToWasClusterMapper earToClusterMapper = new EarToWasClusterMapper(change);
		EarToWasServerMapper earToServerMapper = new EarToWasServerMapper(change);
		EarToWasUnmanagedServerMapper earToWasUnmanagedServerMapper = new EarToWasUnmanagedServerMapper(change);
		WarToWasClusterMapper warToClusterMapper = new WarToWasClusterMapper(change);
		WarToWasServerMapper warToServerMapper = new WarToWasServerMapper(change);
		WarToWasUnmanagedServerMapper warToUnmanagedServerMapper = new WarToWasUnmanagedServerMapper(change);
		EjbJarToWasTargetMapper ejbJarMapper = new EjbJarToWasTargetMapper(change);
		StaticContentToApacheHttpdServerMapper staticContentMapper = new StaticContentToApacheHttpdServerMapper(change);
		WasDataSourceToWasTargetMapper dsToClusterMapper = new WasDataSourceToWasTargetMapper(change);
		WasWmqQueueToWasTargetMapper qToClusterMapper = new WasWmqQueueToWasTargetMapper(change);
		WasWmqQueueConnectionFactoryToWasTargetMapper qcfToClusterMapper = new WasWmqQueueConnectionFactoryToWasTargetMapper(change);
		WasWmqTopicToWasTargetMapper tToServerMapper = new WasWmqTopicToWasTargetMapper(change);
		WasWmqTopicConnectionFactoryToWasTargetMapper tcfToClusterMapper = new WasWmqTopicConnectionFactoryToWasTargetMapper(change);
		WasJndiPropertiesToWasTargetMapper jndiPropertiesToClusterMapper = new WasJndiPropertiesToWasTargetMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);
		LibrariesToHostMapper librariesToHostMapper = new LibrariesToHostMapper(change);
		SharedLibraryToWasScopeTargetMapper sharedLibToWasScopeTargetMapper = new SharedLibraryToWasScopeTargetMapper(change);
		SqlFolderToDatabaseMapper sqlFoldermapper = new SqlFolderToDatabaseMapper(change);
		SqlScriptToDatabaseMapper sqlScriptMapper = new SqlScriptToDatabaseMapper(change);
		GenericFolderToHostMapper genericFolderToHostMapper = new GenericFolderToHostMapper(change);

		affectedWasTargets = getAffectedWasTargets(earToClusterMapper, earToServerMapper, earToWasUnmanagedServerMapper, warToClusterMapper, warToServerMapper,
				warToUnmanagedServerMapper, ejbJarMapper, dsToClusterMapper, qToClusterMapper, qcfToClusterMapper, tToServerMapper, tcfToClusterMapper,
				jndiPropertiesToClusterMapper, sharedLibToWasScopeTargetMapper);

		if (affectedWasTargets.isEmpty()) {
			return;
		}

		// get affected Nodes
		Set<WasNodeAgent> affectedWasNodes = new HashSet<WasNodeAgent>();
		for (WasScope wasTarget : affectedWasTargets) {
			if (wasTarget instanceof WasTarget) {
				affectedWasNodes.addAll(((WasTarget) wasTarget).getNodes());
			} else if (wasTarget instanceof WasNodeAgent) {
				affectedWasNodes.add((WasNodeAgent) wasTarget);
			}
		}

		int lastStepCount = steps.size();

		// deletion steps
		Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServerToDestroy = new HashMap<ApacheHttpdServer, Set<String>>();

		earToClusterMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		earToClusterMapper.generateDeletionSteps(steps);
		earToServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		earToServerMapper.generateDeletionSteps(steps);
		earToWasUnmanagedServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		earToWasUnmanagedServerMapper.generateDeletionSteps(steps);

		warToClusterMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		warToClusterMapper.generateDeletionSteps(steps);
		warToServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		warToServerMapper.generateDeletionSteps(steps);
		warToUnmanagedServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToDestroy);
		warToUnmanagedServerMapper.generateDeletionSteps(steps);

		ejbJarMapper.generateDeletionSteps(steps);

		if (steps.size() > lastStepCount) {
			synchronizeNodes(affectedWasNodes, steps);
		}

		// destroy resources
		lastStepCount = steps.size();
		tToServerMapper.generateDeletionSteps(steps);
		tcfToClusterMapper.generateDeletionSteps(steps);
		qToClusterMapper.generateDeletionSteps(steps);
		qcfToClusterMapper.generateDeletionSteps(steps);
		dsToClusterMapper.generateDeletionSteps(steps);
		jndiPropertiesToClusterMapper.generateDeletionSteps(steps);

		sharedLibToWasScopeTargetMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateDeletionSteps(steps);
		librariesToHostMapper.generateDeletionSteps(steps);
		genericFolderToHostMapper.generateDeletionSteps(steps);

		genericFolderToHostMapper.generateAdditionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);
		librariesToHostMapper.generateAdditionSteps(steps);
		sharedLibToWasScopeTargetMapper.generateAdditionSteps(steps);
		sharedLibToWasScopeTargetMapper.generateModificationSteps(steps);

		if (steps.size() > lastStepCount) {
			synchronizeNodes(affectedWasNodes, steps);
		}

		WasClusterRunBook wasClusterRunBook = new WasClusterRunBook();
		Set<Change<WasCluster>> clusterChanges = findClusterChangesForDeploymentChange(change, changePlan);
		for (Change<WasCluster> eachClusterChange : clusterChanges) {
			if (eachClusterChange.isModification()) {
				wasClusterRunBook.getStepsForModifiedCluster(steps, eachClusterChange.getOldRevision(), eachClusterChange.getNewRevision(), changePlan);
			} else if (eachClusterChange.isAddition()) {
				throw new ResolutionException(
						"Cannot deploy an application to a Cluster which needs to be created. Please create the cluster first before deploying to it.");
			} else if (eachClusterChange.isDeletion()) {
				throw new ResolutionException(
						"Cannot deploy an application to a Cluster which will be deleted. Please point the deployment to another cluster.");
			}
		}

		WasServerRunBook wasServerRunBook = new WasServerRunBook();
		Set<Change<WasManagedServer>> serverChanges = findServerChangesForDeploymentChange(change, changePlan);
		for (Change<WasManagedServer> eachServerChange : serverChanges) {
			wasServerRunBook.getStepsForModifiedServer(steps, eachServerChange.getOldRevision(), eachServerChange.getNewRevision());
		}

		// create resources
		lastStepCount = steps.size();

		sqlFoldermapper.generateAdditionSteps(steps);
		sqlScriptMapper.generateAdditionSteps(steps);

		Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServerToCreate = new HashMap<ApacheHttpdServer, Set<String>>();

		jndiPropertiesToClusterMapper.generateAdditionSteps(steps);
		dsToClusterMapper.generateAdditionSteps(steps);
		qcfToClusterMapper.generateAdditionSteps(steps);
		qToClusterMapper.generateAdditionSteps(steps);
		tcfToClusterMapper.generateAdditionSteps(steps);
		tToServerMapper.generateAdditionSteps(steps);

		if (steps.size() > lastStepCount) {
			synchronizeNodes(affectedWasNodes, steps);
		}

		// deploy apps
		lastStepCount = steps.size();

		ejbJarMapper.generateAdditionSteps(steps);
		warToClusterMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		warToClusterMapper.generateAdditionSteps(steps);
		warToServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		warToServerMapper.generateAdditionSteps(steps);
		warToUnmanagedServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		warToUnmanagedServerMapper.generateAdditionSteps(steps);

		earToClusterMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		earToClusterMapper.generateAdditionSteps(steps);
		earToServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		earToServerMapper.generateAdditionSteps(steps);
		earToWasUnmanagedServerMapper.setVirtualHostsPerWebServerCollector(virtualHostsPerWebServerToCreate);
		earToWasUnmanagedServerMapper.generateAdditionSteps(steps);

		if (steps.size() > lastStepCount) {
			synchronizeNodes(affectedWasNodes, steps);
		}

		// modification steps
		earToClusterMapper.generateModificationSteps(steps);
		warToClusterMapper.generateModificationSteps(steps);
		earToServerMapper.generateModificationSteps(steps);
		warToServerMapper.generateModificationSteps(steps);
		warToUnmanagedServerMapper.generateModificationSteps(steps);
		earToWasUnmanagedServerMapper.generateModificationSteps(steps);

		// start apps
		ejbJarMapper.generateStartSteps(steps);
		warToClusterMapper.generateStartSteps(steps);
		warToServerMapper.generateStartSteps(steps);
		warToUnmanagedServerMapper.generateStartSteps(steps);
		earToClusterMapper.generateStartSteps(steps);
		earToServerMapper.generateStartSteps(steps);
		earToWasUnmanagedServerMapper.generateStartSteps(steps);

		//generate plugin and copy
		earToClusterMapper.generateAndCopyPluginSteps(steps);
		warToClusterMapper.generateAndCopyPluginSteps(steps);
		earToServerMapper.generateAndCopyPluginSteps(steps);
		warToServerMapper.generateAndCopyPluginSteps(steps);
		warToUnmanagedServerMapper.generateAndCopyPluginSteps(steps);
		earToWasUnmanagedServerMapper.generateAndCopyPluginSteps(steps);

		// no need to synchronize, as it is handled from generateModificationSteps

		// handle apache and static content
		staticContentMapper.generateDeletionSteps(steps);
		destroyApacheVirtualHosts(virtualHostsPerWebServerToDestroy, steps);
		restartApacheWebServers(virtualHostsPerWebServerToDestroy, steps);

		createApacheVirtualHosts(virtualHostsPerWebServerToCreate, steps);
		staticContentMapper.generateAdditionSteps(steps);
		restartApacheWebServers(virtualHostsPerWebServerToCreate, steps);

	}

	private void synchronizeNodes(Set<WasNodeAgent> affectedWasNodes, List<Step> steps) {
		for (WasNodeAgent node : affectedWasNodes) {
			steps.add(new SynchronizeWasNodeStep(node));
		}
	}

	@SuppressWarnings({"rawtypes", "unchecked"})
	private Set<WasScope> getAffectedWasTargets(Mapper... mappers) {
		Set<WasScope> affectedWasTargets = new HashSet<WasScope>();
		for (Mapper m : mappers) {
			affectedWasTargets.addAll(m.getAffectedTargets());
		}

		return affectedWasTargets;
	}

	@SuppressWarnings("unchecked")
	private Set<Change<WasCluster>> findClusterChangesForDeploymentChange(Change<Deployment> deploymentChange, ChangePlan changePlan) {
		Set<Change<WasCluster>> clusterChanges = new HashSet<Change<WasCluster>>();
		for (Change<?> change : changePlan.getChanges()) {
			if (change.getConfigurationItemClass() == WasCluster.class) {
				Change<WasCluster> clusterChange = (Change<WasCluster>) change;
				if (isClusterDeploymentTarget(clusterChange.getOldRevision(), deploymentChange.getOldRevision())
						|| isClusterDeploymentTarget(clusterChange.getNewRevision(), deploymentChange.getNewRevision())
						|| isClusterDeploymentTarget(clusterChange.getNewRevision(), deploymentChange.getOldRevision())
						|| isClusterDeploymentTarget(clusterChange.getOldRevision(), deploymentChange.getNewRevision())) {
					clusterChanges.add(clusterChange);
				}
			}
		}
		return clusterChanges;
	}

	@SuppressWarnings("unchecked")
	private Set<Change<WasManagedServer>> findServerChangesForDeploymentChange(Change<Deployment> deploymentChange, ChangePlan changePlan) {
		Set<Change<WasManagedServer>> serverChanges = new HashSet<Change<WasManagedServer>>();
		for (Change<?> change : changePlan.getChanges()) {
			if (change.isModification() && change.getConfigurationItemClass() == WasManagedServer.class) {
				Change<WasManagedServer> serverChange = (Change<WasManagedServer>) change;
				if (isServerDeploymentTarget(serverChange.getOldRevision(), deploymentChange.getOldRevision())
						&& isServerDeploymentTarget(serverChange.getNewRevision(), deploymentChange.getNewRevision())) {
					serverChanges.add(serverChange);
				}
			}
		}
		return serverChanges;
	}

	@SuppressWarnings("rawtypes")
	private boolean isClusterDeploymentTarget(WasCluster cluster, Deployment deployment) {
		if (cluster != null && deployment != null) {
			List<DeployableArtifactMapping> deployableArtifactMappings = deployment.getMappingsOfType(DeployableArtifactMapping.class);
			for (DeployableArtifactMapping eachMapping : deployableArtifactMappings) {
				if (eachMapping.getTarget().equals(cluster)) {
					return true;
				}
			}
		}
		return false;
	}

	@SuppressWarnings({"rawtypes"})
	private boolean isServerDeploymentTarget(WasManagedServer server, Deployment deployment) {
		List<DeployableArtifactMapping> deployableArtifactMappings = deployment.getMappingsOfType(DeployableArtifactMapping.class);
		for (DeployableArtifactMapping eachMapping : deployableArtifactMappings) {
			if (eachMapping.getTarget().equals(server)) {
				return true;
			}
		}
		return false;
	}

	private void createApacheVirtualHosts(Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServer, List<Step> steps) {
		for (Map.Entry<ApacheHttpdServer, Set<String>> eachVirtualHostPerWebServer : virtualHostsPerWebServer.entrySet()) {
			ApacheHttpdServer eachWebServer = eachVirtualHostPerWebServer.getKey();
			for (String eachVirtualHost : eachVirtualHostPerWebServer.getValue()) {
				steps.add(new CreateApacheHttpdVirtualHostStep(eachWebServer, eachVirtualHost, Collections.singletonList(eachWebServer)));
			}
		}
	}

	private void destroyApacheVirtualHosts(Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServer, List<Step> steps) {
		for (Map.Entry<ApacheHttpdServer, Set<String>> eachVirtualHostPerWebServer : virtualHostsPerWebServer.entrySet()) {
			ApacheHttpdServer eachWebServer = eachVirtualHostPerWebServer.getKey();
			for (String eachVirtualHost : eachVirtualHostPerWebServer.getValue()) {
				steps.add(new DestroyApacheHttpdVirtualHostStep(eachWebServer, eachVirtualHost));
			}
		}
	}

	private void restartApacheWebServers(Map<ApacheHttpdServer, Set<String>> virtualHostsPerWebServer, List<Step> steps) {
		for (ApacheHttpdServer eachWebServer : virtualHostsPerWebServer.keySet()) {
			steps.add(new RestartApacheHttpdServerStep(eachWebServer));
		}
	}

}
