/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.mapper;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.plugin.was.ci.WarsWebserversVirtualHostMapping;
import com.xebialabs.deployit.plugin.was.ci.WasEarMapping;
import com.xebialabs.deployit.plugin.was.ci.WasFileServing;
import com.xebialabs.deployit.plugin.was.ci.WasManagedApacheHttpdServer;
import com.xebialabs.deployit.plugin.was.ci.WasUnmanagedServer;
import com.xebialabs.deployit.plugin.was.step.CreateWasVirtualHostStep;
import com.xebialabs.deployit.plugin.was.step.DestroyWasVirtualHostStep;
import com.xebialabs.deployit.plugin.was.step.StartWasUnmanagedServerStep;
import com.xebialabs.deployit.plugin.was.step.StopWasUnmanagedServerStep;
import com.xebialabs.deployit.plugin.was.step.UpdateWasFileServingEnabledStep;
import com.xebialabs.deployit.plugin.was.step.UpdateWasVirtualHostAliasesStep;
import com.xebialabs.deployit.plugin.was.step.WasDeployApplicationStep;
import com.xebialabs.deployit.plugin.was.step.WasStartApplicationStep;
import com.xebialabs.deployit.plugin.was.step.WasStopApplicationStep;
import com.xebialabs.deployit.plugin.was.step.WasUndeployApplicationStep;

public class EarToWasUnmanagedServerMapper extends JeeWebArtifactToWasTargetMapper<Ear, WasEarMapping, WasUnmanagedServer> {

	public EarToWasUnmanagedServerMapper(Change<Deployment> change) {
		super(change, false);
	}

	@Override
	protected final void generateAdditionStepsForAddedMapping(Ear ear, WasEarMapping mapping, WasUnmanagedServer target, List<Step> steps) {
		Set<WasManagedApacheHttpdServer> webservers = getWebservers(mapping);

		String virtualHostAlias = validateAndGetVirtualHostFromMapping(mapping);
		boolean isAppVirtualHostRequired = getAppVirtualHostRequired(mapping);
		if (isAppVirtualHostRequired) {
			steps.add(new CreateWasVirtualHostStep(target.getCell(), ear.getName()));
			steps.add(new UpdateWasVirtualHostAliasesStep(target.getCell(), ear.getName(), virtualHostAlias));
		}
		// for each Wars inside the Ear, create virtual host and update the aliases
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelVirtualHostMappingPresent()) {
				steps.add(new CreateWasVirtualHostStep(target.getCell(),  getVirtualHostFromMappingOrDefault(eachMapping, ear.getName())));
				steps.add(new UpdateWasVirtualHostAliasesStep(target.getCell(),  getVirtualHostFromMappingOrDefault(eachMapping, ear.getName()), getVirtualHostAliasFromMappingOrDefault(eachMapping, virtualHostAlias)));
			}
		}
		steps.add(new StopWasUnmanagedServerStep(target));
		steps.add(new StartWasUnmanagedServerStep(target));
		
		generateDeployStep(ear, mapping, target, webservers, steps);
		
		// for each Wars inside the Ear , Update File Serving Enabled.
		List<WarsWebserversVirtualHostMapping> fileServingEnabledPerWars = getFileServingEnabledPerWars(mapping);
		for (WarsWebserversVirtualHostMapping warsWebserversVHMapping : fileServingEnabledPerWars) {
			if (warsWebserversVHMapping.getFileServing() != null && warsWebserversVHMapping.getFileServing() != WasFileServing.DO_NOT_OVERRIDE) {
				steps.add(new UpdateWasFileServingEnabledStep(ear, target, warsWebserversVHMapping.getWarName(), warsWebserversVHMapping.getFileServing()));
			}
		}
		
		if (isAppVirtualHostRequired) {
			for (WasManagedApacheHttpdServer eachWebServer : webservers) {
				associateVirtualHostWithWebServer(virtualHostsPerWebServer, eachWebServer, validateAndGetVirtualHostFromMapping(mapping));
			}
		}
		
		// for each War present in the Ear ,associate VirtualHost With WebServer
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getNewRevision().getTarget(),
				        eachMapping.getWebserver());
				associateVirtualHostWithWebServers(webServersFromEnvironment, getVirtualHostAliasFromMappingOrDefault(eachMapping, virtualHostAlias));
			}
		}

	}

	@Override
	protected final void generateDeletionStepsForDeletedMapping(Ear ear, WasEarMapping mapping, WasUnmanagedServer target, List<Step> steps) {
		Set<WasManagedApacheHttpdServer> webservers = getWebservers(mapping);

		generateStopStep(ear, target, mapping, steps);
		generateUndeployStep(ear, target, mapping, steps);
		boolean isAppVirtualHostRequired = getAppVirtualHostRequired(mapping);
		if (isAppVirtualHostRequired) {
			steps.add(new DestroyWasVirtualHostStep(target.getCell(), ear.getName()));
		}
		steps.add(new StopWasUnmanagedServerStep(target));
		steps.add(new StartWasUnmanagedServerStep(target));

		// for each War present in the Ear file ,delete virtual host 
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelVirtualHostMappingPresent()) {
				steps.add(new DestroyWasVirtualHostStep(target.getCell(),  getVirtualHostFromMappingOrDefault(eachMapping, ear.getName())));
			}
		}
	
		String virtualHostAlias = validateAndGetVirtualHostFromMapping(mapping);
		if (isAppVirtualHostRequired) {
			for (WasManagedApacheHttpdServer eachWebServer : webservers) {
				associateVirtualHostWithWebServer(virtualHostsPerWebServer, eachWebServer, virtualHostAlias);
			}
		}
		
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getOldRevision().getTarget(),
				        eachMapping.getWebserver());
				associateVirtualHostWithWebServers(webServersFromEnvironment, getVirtualHostAliasFromMappingOrDefault(eachMapping, virtualHostAlias));
			}
		}
	}

	@Override
	protected void generateModificationStepsForModifiedMapping(Ear oldMappingSource, WasEarMapping oldVersionOfModifiedMapping,
	        WasUnmanagedServer oldMappingTarget, Ear newMappingSource, WasEarMapping newVersionOfModifiedMapping, WasUnmanagedServer newMappingTarget,
	        List<Step> steps) {

		boolean mappingSourceChanged = !oldMappingSource.equals(newMappingSource);
		Set<WasManagedApacheHttpdServer> webservers = getWebservers(newVersionOfModifiedMapping);

		if (mappingSourceChanged) {
			generateStopStep(oldMappingSource, oldMappingTarget,oldVersionOfModifiedMapping, steps);
			generateUndeployStep(oldMappingSource, oldMappingTarget,oldVersionOfModifiedMapping, steps);
		}

		if (!oldVersionOfModifiedMapping.getVirtualHost().equals(newVersionOfModifiedMapping.getVirtualHost())) {
			steps.add(new StopWasUnmanagedServerStep(oldMappingTarget));
			steps.add(new DestroyWasVirtualHostStep(oldMappingTarget.getCell(), oldMappingSource.getName()));
			steps.add(new CreateWasVirtualHostStep(newMappingTarget.getCell(), newMappingSource.getName()));
			steps.add(new UpdateWasVirtualHostAliasesStep(newMappingTarget.getCell(), newMappingSource.getName(),
			        validateAndGetVirtualHostFromMapping(newVersionOfModifiedMapping)));
			steps.add(new StartWasUnmanagedServerStep(newMappingTarget));
		}

		if (mappingSourceChanged) {
			generateDeployStep(newMappingSource, newVersionOfModifiedMapping, newMappingTarget, webservers, steps);
			// for each Wars inside the Ear , Update File Serving Enabled.
			List<WarsWebserversVirtualHostMapping> fileServingEnabledPerWars = getFileServingEnabledPerWars(newVersionOfModifiedMapping);
			for (WarsWebserversVirtualHostMapping warsWebserversVHMapping : fileServingEnabledPerWars) {
				if (warsWebserversVHMapping.getFileServing() != null && warsWebserversVHMapping.getFileServing() != WasFileServing.DO_NOT_OVERRIDE) {
					steps.add(new UpdateWasFileServingEnabledStep(newMappingSource, newMappingTarget, warsWebserversVHMapping.getWarName(), warsWebserversVHMapping.getFileServing()));
				}
			}
		}
	}

	@Override
	protected void generateDeployStep(Ear artifact, WasEarMapping mapping, WasUnmanagedServer server, Collection<WasManagedApacheHttpdServer> webservers,
	        List<Step> steps) {
		// vhost name == artifact's name (see generateAdditionStepsForAddedMapping in JeeArtifactToWasClusterMapper)
		steps.add(new WasDeployApplicationStep(server.getCell(), artifact, Collections.singleton(server), webservers, artifact.getName(), mapping
		        .getSharedLibraries(), mapping.getStartingWeight(), mapping.getClassLoaderMode(), mapping.getClassLoaderPolicy(), mapping
		        .getWarClassLoaderMapping(), mapping.getSecurityRoleUserGroupMappings(),getUpdatedWebserversMapping(mapping)));
	}

	@Override
	protected void generateUndeployStep(Ear artifact, WasUnmanagedServer server, WasEarMapping mapping, List<Step> steps) {
		steps.add(new WasUndeployApplicationStep(server.getCell(), mapping.getArtifact()));
	}

	@Override
	protected void generateStartStep(Ear artifact, WasUnmanagedServer server, WasEarMapping mapping, List<Step> steps) {
		steps.add(new WasStartApplicationStep(server, mapping.getArtifact()));
	}

	@Override
	protected void generateStopStep(Ear artifact, WasUnmanagedServer server, WasEarMapping mapping, List<Step> steps) {
		steps.add(new WasStopApplicationStep(server, mapping.getArtifact()));
	}

	@Override
	protected String getVirtualHostFromMapping(WasEarMapping mapping) {
		return mapping.getVirtualHost();
	}

	@Override
	public void setDefaults(Deployment d, WasEarMapping m) {
		m.setVirtualHost(d.getVhostDefinition());
	}

	@Override
    protected List<WarsWebserversVirtualHostMapping> getWebserversAndVirtualHostPerWar(WasEarMapping mapping) {
	    return mapping.getWarsWebserversVHMapping();
    }
	
	@Override
    protected List<WarsWebserversVirtualHostMapping> getFileServingEnabledPerWars(WasEarMapping mapping) {
	    return getWebserversAndVirtualHostPerWar(mapping);
    }

}
