/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.step;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.Collection;
import java.util.HashSet;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.EjbJar;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.hostsession.HostSessionFactory;
import com.xebialabs.deployit.plugin.was.ci.WasCell;
import com.xebialabs.deployit.plugin.was.ci.WasCluster;
import com.xebialabs.deployit.plugin.was.ci.WasManagedApacheHttpdServer;
import com.xebialabs.deployit.plugin.was.ci.WasServer;
import com.xebialabs.deployit.util.ExtendedStringUtils;

/**
 * Deploys an {@link Ear}, a {@link War} or an {@link EjbJar} to a Collection of {@link WasCluster} in a {@link WasCell}
 */
@SuppressWarnings("serial")
public class WasDeployApplicationOnServersStep extends WasStepBase {

	protected static final String DEPLOY_APPLICATION_SCRIPT_RESOURCE_PATH = STEP_RESOURCES_PATH + "deploy-application.py";

	protected NamedDeployableArtifact application;

	protected String extension;

	protected WasCell cell;

	protected Collection<WasServer> servers;

	protected Collection<WasManagedApacheHttpdServer> webservers;

	protected String virtualHostName;

	protected String contextRoot;
	
	protected int startingWeight = 1;

	public WasDeployApplicationOnServersStep(WasCell cell, Ear ear, Collection<WasServer> servers, Collection<WasManagedApacheHttpdServer> webservers,
			String virtualHostName, int startingWeight) {
		super(cell);

		init(cell, servers, webservers, virtualHostName);
		this.application = checkNotNull(ear);
		this.extension = ".ear";
		this.contextRoot = "";
		this.startingWeight = startingWeight;

		validate();
		initDescription();
	}
	

	public WasDeployApplicationOnServersStep(WasCell cell, War war, Collection<WasServer> servers, Collection<WasManagedApacheHttpdServer> webservers,
			String virtualHostName, String contextRoot, int startingWeight) {
		super(cell);

		init(cell, servers, webservers, virtualHostName);
		this.application = checkNotNull(war);
		this.extension = ".war";
		if (contextRoot == null) {
			throw new ResolutionException("Context root definition \"" + contextRoot + "\" is not valid. It is required for a war deployment.");
		}
		this.contextRoot = contextRoot;
		this.startingWeight = startingWeight;

		validate();
		initDescription();
	}

	public WasDeployApplicationOnServersStep(WasCell cell, EjbJar ejbJar, Collection<WasServer> servers, Collection<WasManagedApacheHttpdServer> webservers, int startingWeight) {
		super(cell);

		init(cell, servers, webservers, null);
		this.application = checkNotNull(ejbJar);
		this.extension = ".jar";
		this.contextRoot = "";
		this.startingWeight = startingWeight;

		validate();
		initDescription();
	}

	private void init(WasCell cell, Collection<WasServer> servers, Collection<WasManagedApacheHttpdServer> webservers, String virtualHostName) {
		this.cell = checkNotNull(cell);
		this.servers = checkNotNull(servers);
		//Make a copy of the Set because it is a EnhancedByCGLib proxy and we lose it when serializing the step.
		this.webservers = new HashSet<WasManagedApacheHttpdServer>(webservers);
		this.virtualHostName = StringUtils.defaultString(virtualHostName, "default_host");
	}

	private void validate() {
		if (servers.isEmpty()) {
			throw new IllegalArgumentException("Collection of servers is empty");
		}

		for (WasServer eachServer : servers) {
			if (!eachServer.getNode().getCell().equals(cell)) {
				throw new IllegalArgumentException("Server " + eachServer.getLabel() + " is part of cell " + eachServer.getNode().getCell()
						+ " so it cannot be targeted when deploying to cell " + cell.getLabel());
			}
		}

		for (WasManagedApacheHttpdServer eachWebServer : webservers) {
			if (!eachWebServer.getNode().getCell().equals(cell)) {
				throw new IllegalArgumentException("WebServer " + eachWebServer.getLabel() + " is part of cell " + eachWebServer.getNode().getCell()
						+ " so it cannot be targeted when deploying to cell " + cell.getLabel());
			}
		}
	}

	private void initDescription() {
		StringBuilder desc = new StringBuilder();
		desc.append("Deploy application ");
		desc.append(ExtendedStringUtils.getQuotedString(application.getLabel()));
		if (servers.size() == 1) {
			desc.append(" on server ");
			desc.append(ExtendedStringUtils.getQuotedString(servers.iterator().next().getLabel()));
		} else {
			desc.append(" on multiple servers ");
		}
		if (webservers.size() == 1) {
			desc.append(" and webserver " + ExtendedStringUtils.getQuotedString(webservers.iterator().next().getLabel()));
		} else if (webservers.size() > 1) {
			desc.append(" and multiple webservers ");
		}
		desc.append(" in cell ");
		desc.append(ExtendedStringUtils.getQuotedString(cell.getLabel()));
		desc.append(" with virtual host ");
		desc.append(virtualHostName);
		setDescription(desc.toString());
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession rhs = cell.connectToAdminHost();
		try {
			HostFile uploadedApplicationFile = uploadApplication(rhs);

			String earFilePath = uploadedApplicationFile.getPath().replace('\\', '/');
			StringBuilder targets = new StringBuilder();
			for (WasServer eachServer : servers) {
				if (targets.length() > 0) {
					targets.append("+");
				}
				targets.append("WebSphere:cell=");
				targets.append(cell.getName());
				targets.append(",server=");
				targets.append(eachServer.getName());
			}
			for (WasManagedApacheHttpdServer eachWebserver : webservers) {
				targets.append("+WebSphere:cell=");
				targets.append(cell.getName());
				targets.append(",node=");
				targets.append(eachWebserver.getNode().getName());
				targets.append(",server=");
				targets.append(eachWebserver.getName());
			}
			int res = executeWsadminJythonScriptResource(ctx, rhs, DEPLOY_APPLICATION_SCRIPT_RESOURCE_PATH, application.getName(), earFilePath, targets.toString(), virtualHostName,
					contextRoot, Integer.toString(startingWeight));
			return res == 0;
		} finally {
			rhs.close();
		}
	}

	private HostFile uploadApplication(HostSession rhs) {
		HostFile uploadedApplicationFile;
		HostSession lhs = HostSessionFactory.getHostSession(Host.getLocalHost());
		HostFile applicationFileToUpload = lhs.getFile(application.getLocation());
		try {
			uploadedApplicationFile = rhs.getTempFile(application.getName(), extension);
			HostFileUtils.copy(applicationFileToUpload, uploadedApplicationFile);
		} finally {
			lhs.close();
		}
		return uploadedApplicationFile;
	}

	public NamedDeployableArtifact getApplication() {
		return application;
	}

	public String getExtension() {
		return extension;
	}

	public WasCell getCell() {
		return cell;
	}

	public Collection<WasManagedApacheHttpdServer> getWebservers() {
		return webservers;
	}

	public String getVirtualHostName() {
		return virtualHostName;
	}

	public String getContextRoot() {
		return contextRoot;
	}

}
