package com.xebialabs.deployit.plugin.api.deployment.planning;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;

import java.util.Collection;

/**
 * Context that is passed along during the Planning stage of a Deployment. This can be used to add {@link DeploymentStep} to the current plan,
 * or store and retrieve attributes to transfer knowledge between multiple Contributors/Processors.
 */
public interface DeploymentPlanningContext {

    /**
     * Add a step to the current plan.
     * @param step the step to add.
     * @deprecated Use Step instead of DeploymentStep
     */
    void addStep(DeploymentStep step);

    /**
     * Add multiple steps to the current plan.
     * @param steps the steps to add.
     * @deprecated Use Step instead of DeploymentStep
     */
    void addSteps(DeploymentStep... steps);

    /**
     * Add multiple steps to the current plan.
     * @param steps the steps to add.
     * @deprecated Use Step instead of DeploymentStep
     */
    void addSteps(Collection<DeploymentStep> steps);

    /**
     * Add a step to the current plan.
     * @param step the step to add.
     */
    void addStep(Step step);

    /**
     * Add multiple steps to the current plan.
     * @param steps the steps to add.
     */
    void addSteps(Step... steps);

    /**
     * Add multiple steps to the current plan.
     * @param steps the steps to add.
     */
    void addSteps(Iterable<Step> steps);

    /**
     * Add a checkpoint so that the state after this step has been executed can be committed.
     * <p/>
     * <em>N.B.</em> This does not add the step to the planning context, use {@link #addStep(com.xebialabs.deployit.plugin.api.flow.Step)}.
     * @param step The Step that needs to complete before we checkpoint
     * @param delta The Delta that needs to be checkpointed.
     */
    void addCheckpoint(Step step, Delta delta);

    /**
     * Add a checkpoint so that the state after this step has been executed can be committed.
     * <p/>
     * <em>N.B.</em> This does not add the step to the planning context, use {@link #addStep(com.xebialabs.deployit.plugin.api.flow.Step)}.
     * @param step The Step that needs to complete before we checkpoint
     * @param delta The Delta that needs to be checkpointed.
     * @param overrideOperation The operation that is actually performed on the Delta, useful in case of split implementations of Update Delta's (ie. DESTROY -> CREATE).
     */
    void addCheckpoint(Step step, Delta delta, Operation overrideOperation);

    /**
     * Add a checkpoint so that the state after this step has been executed can be committed.
     * <p/>
     * <em>N.B.</em> This does not add the step to the planning context, use {@link #addStep(com.xebialabs.deployit.plugin.api.flow.Step)}.
     * @param step The Step that needs to complete before we checkpoint
     * @param deltas The Deltas that need to be checkpointed.
     */
    void addCheckpoint(Step step, Iterable<Delta> deltas);

    /**
     * Add a step and register a checkpoint so that the state after this step has been executed can be committed.
     * <p/>
     * @param step The Step that needs to complete before we checkpoint
     * @param delta The Delta that needs to be checkpointed.
     */
    void addStepWithCheckpoint(Step step, Delta delta);

    /**
     * Add a step and register a checkpoint so that the state after this step has been executed can be committed.
     * <p/>
     * @param step The Step that needs to complete before we checkpoint
     * @param delta The Delta that needs to be checkpointed.
     * @param overrideOperation The operation that is actually performed on the Delta, useful in case of split implementations of Update Delta's (ie. DESTROY -> CREATE).
     */
    void addStepWithCheckpoint(Step step, Delta delta, Operation overrideOperation);

    /**
     * Add a step and register a checkpoint so that the state after this step has been executed can be committed.
     * <p/>
     * @param step The Step that needs to complete before we checkpoint
     * @param deltas The Deltas that need to be checkpointed.
     */
    void addStepWithCheckpoint(Step step, Iterable<Delta> deltas);

    /**
     * Returns the planning context attribute with the given name, or <tt>null</tt> if there is no attribute by that name.
     * 
     * @param name
     *            the name of the attribute
     * @return the value of the attribute, or <tt>null</tt> if there is no attribute by that name.
     */
    Object getAttribute(String name);

    /**
     * Stores an attribute in the planning context. This attribute will be available while the deployment plan is being created.
     * 
     * @param name
     *            the name of the attribute
     * @param value
     *            the value of the attribute
     */
    void setAttribute(String name, Object value);

    /**
     * Returns the DeployedApplication for which the plan is being created.
     *
     * @return the DeployedApplication for which the plan is being created.
     */
    DeployedApplication getDeployedApplication();

    /**
     * Returns a read-only view of the repository of Deployit.
     *
     * <em>Note:</em> This repository cannot be stored in a step and should only be used during planning.
     *
     * @return a read-only view of the repository
     */
    ReadOnlyRepository getRepository();
}
