package com.xebialabs.deployit.plugin.api.reflect;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.api.reflect.Descriptor.from;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredStringAttribute;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

public class DescriptorRegistry {
	private static final Map<Type, Descriptor> descriptors = newHashMap();

	private static final Multimap<Type, Type> subtypes = HashMultimap.create();

	// N.B.: Logger must be defined at top of file so that other static initializers can access it
	private static Logger logger = LoggerFactory.getLogger(DescriptorRegistry.class);

    static void register(Class<? extends ConfigurationItem> ci) {
        Descriptor descriptor = from(ci);
        logger.debug("Found ConfigurationItem: {}", descriptor);
        register(descriptor);
    }

    static void register(Descriptor descriptor) {
	    checkState(!descriptors.containsKey(descriptor.getType()), "The type [%s] is already registered in Deployit.", descriptor.getType());
        descriptors.put(descriptor.getType(), descriptor);
    }

 	static void registerSyntheticTypeDefinition(Element syntheticElement) {
		Descriptor descriptor = Descriptor.from(syntheticElement);
		logger.debug("Found ConfigurationItem: {}", descriptor);
		register(descriptor);
	}

	static void registerSyntheticTypeModification(Element typeModificationElement) {
		String typeName = getRequiredStringAttribute(typeModificationElement, "type");
		Descriptor d = getDescriptor(typeName);
		d.parseTypeModification(typeModificationElement);
	}

	static void registerSubtype(Type supertype, Type subtype) {
		checkState(!supertype.equals(subtype), "Type %s cannot be registered as its own subtype", supertype);
		subtypes.put(supertype, subtype);
	}

	public static Collection<Descriptor> getDescriptors() {
		return Collections.unmodifiableCollection(descriptors.values());
	}

	public static Collection<Type> getSubtypes(Type supertype) {
		return Collections.unmodifiableCollection(subtypes.get(supertype));
	}

	public static Descriptor getDescriptor(String prefixedName) {
		return getDescriptor(Type.valueOf(prefixedName));
	}

	public static Descriptor getDescriptor(String prefix, String name) {
		return getDescriptor(Type.valueOf(prefix, name));
	}

	public static Descriptor getDescriptor(Type type) {
		checkState(!descriptors.isEmpty(), "DescriptorRegistry not booted");
		checkArgument(exists(type), "Unknown type [%s]", type);
		return descriptors.get(type);
	}

    public static boolean exists(Type type) {
        return descriptors.containsKey(type);
    }
}
