package com.xebialabs.deployit.plugin.api.udm.artifact;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static java.lang.Math.abs;
import static java.util.regex.Pattern.CASE_INSENSITIVE;
import static java.util.regex.Pattern.COMMENTS;

import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.collect.MapMaker;
import com.google.common.collect.Sets;
import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.api.utils.TFileUtils;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalFile;

import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileReader;
import de.schlichtherle.truezip.file.TFileWriter;

public class Artifacts {

	public static Map<String, Pattern> patternMap = new MapMaker().makeComputingMap(new Function<String, Pattern>() {
        public Pattern apply(String regex) {
	        return Pattern.compile(regex, COMMENTS | CASE_INSENSITIVE);
        }
	});

	public static void scanPlaceholders(SourceArtifact artifact, PlaceholderScanner scanner) {
		checkArgument(artifact.getFile() != null, artifact + " has no file");
		checkArgument(artifact.getFile() instanceof LocalFile, "Cannot scan for placeholders in " + artifact + " because its file is a " + artifact.getFile().getClass().getName() + " and not a " + LocalFile.class.getName());
        
		artifact.setPlaceholders(Sets.<String> newTreeSet());
		if (artifact.hasProperty("scanPlaceholders") && !(Boolean) artifact.getProperty("scanPlaceholders")) {
			return;
		}
        TFile from = toTFile(artifact.getFile());
		try {
            doScanPlaceholders(artifact, from, scanner);
		} finally {
			TFileUtils.umountQuietly(from);
		}
	}

	private static void doScanPlaceholders(SourceArtifact artifact, TFile from, PlaceholderScanner scanner) {
		if(shouldExcludeFile(from, artifact.getExcludeFileNamesRegex())) {
            return;
        }

		if (from.isDirectory()) {
	        try {
				for (TFile f : from.listFiles()) {
					doScanPlaceholders(artifact, f, scanner);
				}
	        } finally {
		        TFileUtils.umountQuietly(from);
	        }
		} else if (isTextFile(from, artifact.getTextFileNamesRegex())) {
			artifact.getPlaceholders().addAll(readPlaceholders(from, scanner));
		}
    }

	private static Set<String> readPlaceholders(final TFile from, final PlaceholderScanner scanner) {
		Reader in = null;
		try {
            logger.trace("Reading placeholders from file {}",from.getPath());
			in = new TFileReader(from);
			return scanner.scan(in);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot scan for placeholders in " + from, exc);
		} catch (RuntimeException exc) {
			throw new RuntimeException("Cannot scan for placeholders in " + from, exc);
		} finally {
			Closeables.closeQuietly(in);
		}
	}

	public static void replacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, PlaceholderReplacer replacer) {
		if(derivedArtifact.getSourceArtifact() == null) {
			derivedArtifact.setFile(null);
		} else {
			checkArgument(derivedArtifact.getSourceArtifact().getFile() != null, "%s has no file", derivedArtifact.getSourceArtifact());
			checkArgument(derivedArtifact.getSourceArtifact().getFile() instanceof LocalFile, "Cannot replace placeholders in %s because its file is not a LocalFile but a %s",
			        derivedArtifact.getSourceArtifact(), derivedArtifact.getSourceArtifact().getFile().getClass().getName());

			TFile from = toTFile(derivedArtifact.getSourceArtifact().getFile());
			try {
				boolean isBinaryFile = from.isFile() && !isTextFile(from, getTextFileNamesRegex(derivedArtifact));

				if (derivedArtifact.getPlaceholders().isEmpty() || isBinaryFile) {
					derivedArtifact.setFile(derivedArtifact.getSourceArtifact().getFile());
				} else {
					TFile to = getOutputFile(derivedArtifact);
					try {
						doReplacePlaceholders(derivedArtifact, from, to, replacer);
	
						File deployedFile = saveArchive(to);
	
						derivedArtifact.setFile(LocalFile.valueOf(deployedFile));
					} finally {
						TFileUtils.umountQuietly(to);
					}
				}
			} finally {
				TFileUtils.umountQuietly(from);
			}
		}
	}

	private static void doReplacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, TFile from, TFile to, PlaceholderReplacer replacer) {
		boolean shouldExcludeFile = shouldExcludeFile(from, getExcludeFileNamesRegex(derivedArtifact));
		
		if (!shouldExcludeFile && from.isDirectory()) {
			try {
		        try {
		            to.mkdir(false);
		        } catch(IOException exc) {
		            throw new RuntimeIOException("Cannot create directory " + to, exc);
		        }
	
		        for (TFile f : from.listFiles()) {
					TFile t = new TFile(to, f.getName());
					doReplacePlaceholders(derivedArtifact, f, t, replacer);
				}
			} finally {
				TFileUtils.umountQuietly(from);
				TFileUtils.umountQuietly(to);
			}
		} else if (!shouldExcludeFile && isTextFile(from, getTextFileNamesRegex(derivedArtifact))) {
			replace(from, to, replacer, derivedArtifact.getPlaceholders());
		} else {
			try {
                from.cp_rp(to);
            } catch (IOException exc) {
            	throw new RuntimeIOException("Cannot copy " + from + " to " + to, exc);
            }
		}
	}

	private static void replace(final TFile from, final TFile to, final PlaceholderReplacer replacer, Map<String, String> resolution) {
		Reader reader = null;
		Writer writer = null;
		try {
			reader = new TFileReader(from);
			writer = new TFileWriter(to);
			replacer.replace(reader, writer, resolution);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot copy " + from + " to " + to + " while replacing placeholders", exc);
		} finally {
			Closeables.closeQuietly(reader);
			Closeables.closeQuietly(writer);
		}
	}

	private static final TFile getOutputFile(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
		OverthereFile workDir = derivedArtifact.getSourceArtifact().getFile().getParentFile();

		Random r = new Random();
		String name = derivedArtifact.getName();
		for(;;) {
			OverthereFile deployedArchiveDir = workDir.getFile(name);

			if(!deployedArchiveDir.exists()) {
				deployedArchiveDir.mkdir();
				return toTFile(deployedArchiveDir.getFile(derivedArtifact.getSourceArtifact().getFile().getName()));
			}

			name = derivedArtifact.getName() + abs(r.nextInt());
		}
	}

	private static File saveArchive(TFile outputArchive) {
		if (outputArchive.isArchive() && outputArchive.getEnclArchive() == null && outputArchive.isDirectory()) {
			try {
				TFile.umount(outputArchive);
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot write archive " + outputArchive, exc);
			}
        }

		// Return a regular java.io.File pointing to the file, folder or archive just written
		return new File(outputArchive.getPath());
    }

    private static boolean shouldExcludeFile(final TFile f, final String excludeFileNamesRegex) {
        if (emptyToNull(excludeFileNamesRegex) == null) {
            return false;
        }

        Pattern excludeFileNamesPattern = patternMap.get(excludeFileNamesRegex);
        Matcher excludeFileNamesMatcher = excludeFileNamesPattern.matcher(f.getPath());
        boolean exclude = excludeFileNamesMatcher.matches();
        if(exclude) {
          logger.debug("Excluding file {} from scanning", f);
        }
        return exclude;
    }

    private static String getExcludeFileNamesRegex(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
		return ((SourceArtifact) derivedArtifact.getSourceArtifact()).getExcludeFileNamesRegex();
	}

    private static boolean isTextFile(final TFile f, final String textFileNamesRegex) {
		checkNotNull(textFileNamesRegex, "Regex is null");

		Pattern textFileNamesPattern = patternMap.get(textFileNamesRegex);
		Matcher textFileNamesMatcher = textFileNamesPattern.matcher(f.getName());
		boolean isTextFile = textFileNamesMatcher.matches();
		logger.debug("Determined {} to be a {} file", f.getName(), isTextFile ? "text" : "binary");

		return isTextFile;
	}

	private static String getTextFileNamesRegex(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
		return ((SourceArtifact) derivedArtifact.getSourceArtifact()).getTextFileNamesRegex();
	}

	private static TFile toTFile(OverthereFile file) {
		return new TFile(((LocalFile) file).getFile());
	}

	private static final Logger logger = LoggerFactory.getLogger(Artifacts.class);

}
