package com.xebialabs.deployit.plugin.api.reflect;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Predicates.equalTo;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.ReflectionUtils.searchField;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.childByName;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.forEach;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalBooleanAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalStringAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalTypeAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredStringAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredTypeAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.ValidationRuleConverter.RULE_ELEMENT_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.ValidationRuleConverter.isRule;
import static com.xebialabs.deployit.plugin.api.udm.Deployed.CONTAINER_FIELD;
import static com.xebialabs.deployit.plugin.api.udm.Deployed.DEPLOYABLE_FIELD;
import static java.lang.reflect.Modifier.isAbstract;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.Closure;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Prefix;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;


public class Descriptor {

	static final String PLACEHOLDERS_FIELD = "placeholders";

	private Type type;
	private Class<? extends ConfigurationItem> clazz;
	private String description;
	private Metadata.ConfigurationItemRoot root;
	private List<Type> superclasses = newArrayList();
	private Set<Type> interfaces = newHashSet();
	private boolean virtual = false;

	private Map<String, PropertyDescriptor> properties = newLinkedHashMap();
	private Type deployableType;
	private Type containerType;

	private Type generatedDeployableType;
	private Type generatedDeployableBase;
	private String generatedDeployableDescription;

	private boolean hierarchyInitialized = false;
	private Map<String, MethodDescriptor> controlTasks = newHashMap();
	private List<Validator<ConfigurationItem>> validators = newArrayList();
	private Field syntheticPropertiesField;

	private Descriptor(Class<? extends ConfigurationItem> clazz) {
		this.type = Type.valueOf(clazz);
		this.virtual = clazz.isInterface();
		this.clazz = clazz;
	}

	private Descriptor(Type type) {
		this.type = type;
	}

	static Descriptor from(Class<? extends ConfigurationItem> clazz) {
		try {
			Descriptor descriptor = new Descriptor(clazz);
			descriptor.initMetadata();
			descriptor.scanClass();
			return descriptor;
		} catch (RuntimeException e) {
			throw new DescriptorException("Could not create descriptor for: " + clazz.getName(), e);
		}
	}

	private void initMetadata() {
		if (virtual) {
			return;
		}

        boolean directlyAnnotated = false;
        Annotation[] declaredAnnotations = clazz.getDeclaredAnnotations();
        for (Annotation declaredAnnotation : declaredAnnotations) {
            if (declaredAnnotation.annotationType().equals(Metadata.class)) {
                directlyAnnotated = true;
            }
        }

        Metadata annotation = checkNotNull(clazz.getAnnotation(Metadata.class), "Class " + clazz.getName() + " or one of its ancestors does not have a @Metadata annotation");
		description = annotation.description();
		root = annotation.root();
		virtual = (directlyAnnotated && annotation.virtual()) || Modifier.isAbstract(clazz.getModifiers());
	}

	private void scanClass() {
		findProperties();
		findControlTasks();
		findInterfaces();
		findValidations();

		Class<?> superclass = clazz.getSuperclass();
		if (superclass != null && ConfigurationItem.class.isAssignableFrom(superclass)) {
			Type supertype = Type.valueOf(superclass);
			addSuperClass(supertype);
		}

		initDeployableAndContainerTypes();
	}

	@SuppressWarnings("unchecked")
    private void findValidations() {
		for (Annotation annotation : clazz.getAnnotations()) {
			if (isRule(annotation)) {
				Validator<ConfigurationItem> v = (Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(annotation);
				this.validators.add(v);
			}
		}
	}

	private void findControlTasks() {
		for (Method method : clazz.getDeclaredMethods()) {
			if (method.isAnnotationPresent(ControlTask.class)) {
				MethodDescriptor from = MethodDescriptor.from(this, method);
				addControlTask(from);
			}
		}
	}

	private void findProperties() {
		for (Field field : clazz.getDeclaredFields()) {
			if (field.isAnnotationPresent(Property.class)) {
				PropertyDescriptor propertyDescriptor = PropertyDescriptor.from(this, field);
				addProperty(propertyDescriptor);
			}
		}
	}

	private void findInterfaces() {
        Class<?>[] clazzInterfaces = clazz.getInterfaces();
        List<Class<?>> allInterfacesFound = newArrayList();
        findAllSuperInterfaces(clazzInterfaces, allInterfacesFound);
		for (Class<?> clazzInterface : allInterfacesFound) {
			if (clazzInterface.getPackage().isAnnotationPresent(Prefix.class)) {
				addInterface(Type.valueOf(clazzInterface));
			}
		}
	}

    private void findAllSuperInterfaces(Class<?>[] childInterfaces, List<Class<?>> allInterfacesFound) {
        for (Class<?> childInterface : childInterfaces) {
            allInterfacesFound.add(childInterface);
            findAllSuperInterfaces(childInterface.getInterfaces(), allInterfacesFound);
        }
    }

    private void initDeployableAndContainerTypes() {
		if (Deployed.class.isAssignableFrom(clazz)) {
			@SuppressWarnings({ "unchecked", "rawtypes" })
			List<Class<?>> typeArguments = ClassUtils.getActualTypeArguments((Class<? extends Deployed>) clazz, Deployed.class);
			checkArgument(typeArguments.size() == 2, "Expected exactly a Deployable and a Container, but got %s", typeArguments);

			Class<?> deployableClass = typeArguments.get(0);
			if (deployableClass != null) {
				checkArgument(Deployable.class.isAssignableFrom(deployableClass), "Expected first item to be a deployable");
				deployableType = Type.valueOf(deployableClass);
			} else {
				deployableType = null;
			}

			Class<?> containerClass = typeArguments.get(1);
			if (containerClass != null) {
				checkArgument(Container.class.isAssignableFrom(containerClass), "Expected second item to be a container");
				containerType = Type.valueOf(containerClass);
			} else {
				containerType = null;
			}
		} else {
			deployableType = null;
			containerType = null;
		}
	}

	static Descriptor from(Element typeElement) {
		Type type = getRequiredTypeAttribute(typeElement, "type");

		Descriptor descriptor = new Descriptor(type);
		descriptor.initSynthetic(typeElement);

		return descriptor;
	}

	private void initSynthetic(Element typeElement) {
		description = getOptionalStringAttribute(typeElement, "description", "Description unavailable");
		virtual = getOptionalBooleanAttribute(typeElement, "virtual", false);

		Type superType = getRequiredTypeAttribute(typeElement, "extends");
		addSuperClass(superType);

		parseSyntheticDeployableAndContainerType(typeElement);
		parseTypeModification(typeElement);
		parseValidators(typeElement);
	}

	private void parseValidators(Element typeElement) {
		forEach(childByName(typeElement, equalTo(RULE_ELEMENT_NAME)), new Closure<Element>() {
			@SuppressWarnings("unchecked")
            @Override
			public void call(Element element) {
				validators.add((Validator<ConfigurationItem>) ValidationRuleConverter.makeRule(element, type));
			}
		});
	}

	private void parseSyntheticDeployableAndContainerType(Element typeElement) {
        deployableType = getOptionalTypeAttribute(typeElement, "deployable-type");
        containerType = getOptionalTypeAttribute(typeElement, "container-type");
        Iterator<Element> generateElements = childByName(typeElement, equalTo("generate-deployable"));
        if (generateElements.hasNext()) {
            Element generateDeployable = generateElements.next();
            generatedDeployableType = getRequiredTypeAttribute(generateDeployable, "type");
            generatedDeployableBase = getRequiredTypeAttribute(generateDeployable, "extends");
            generatedDeployableDescription= getOptionalStringAttribute(generateDeployable, "description", "Description unavailable");
        }
	}

    void parseTypeModification(Element element) {
        forEach(childByName(element, equalTo("property")), new Closure<Element>() {
	        @Override
	        public void call(Element element) {
		        String name = getRequiredStringAttribute(element, "name");
		        PropertyDescriptor newDesc = PropertyDescriptor.from(Descriptor.this, element);
		        PropertyDescriptor oldDesc = properties.get(name);
		        if (oldDesc != null) {
			        newDesc.overrideWith(oldDesc);
		        }
		        addProperty(newDesc);
	        }
        });

	    forEach(childByName(element, equalTo("method")), new Closure<Element>() {
		    @Override
		    public void call(Element element) {
			    MethodDescriptor controlTask = MethodDescriptor.from(Descriptor.this, element);
			    verifyNewControlTask(controlTask);
			    addControlTask(controlTask);
		    }
	    });
    }

	private void verifyNewControlTask(MethodDescriptor controlTask) {
		if (controlTasks.containsKey(controlTask.getName())) {
			throw new IllegalStateException("Cannot override existing Control Task [" + controlTask.getName() + "] with a synthetic one.");
		}
	}

	static Descriptor from(Descriptor deployedDescriptor) {
        Descriptor deployableDescriptor = new Descriptor(deployedDescriptor.generatedDeployableType);
        deployableDescriptor.addSuperClass(deployedDescriptor.generatedDeployableBase);
		deployableDescriptor.initDeployableFromDeployed(deployedDescriptor, deployedDescriptor.generatedDeployableDescription);
		return deployableDescriptor;
	}

	private void initDeployableFromDeployed(Descriptor deployedDescriptor, String generatedDeployableDescription) {
		if (generatedDeployableDescription == null || generatedDeployableDescription.equals("Description unavailable")) {
			description = deployedDescriptor.getDescription() + " (deployable)";
		} else {
			description = generatedDeployableDescription;
		}

		for (PropertyDescriptor pd : deployedDescriptor.getPropertyDescriptors()) {
			String name = pd.getName();
			boolean isUdmField = name.equals(DEPLOYABLE_FIELD) || name.equals(CONTAINER_FIELD)
			        || name.equals(PLACEHOLDERS_FIELD);
			final PropertyKind kind = pd.getKind();
			boolean isReferenceField = kind == CI || kind == SET_OF_CI || kind == LIST_OF_CI;
			if (isUdmField || isReferenceField || pd.isHidden()) {
				continue;
			}

			addProperty(PropertyDescriptor.generateDeployableFrom(this, pd));
		}
	}

	void initHierarchy() {
		if (hierarchyInitialized || superclasses.isEmpty()) {
			return;
		}

		Type toInitFrom = superclasses.get(0);

		do {
			Descriptor superDesc = DescriptorRegistry.getDescriptor(toInitFrom);
			if(superDesc == null) {
				throw new IllegalStateException("Cannot build type hierarchy for " + getType() + " because one of its supertypes cannot be found: " + toInitFrom + " not found");
			}

			// For synthetic types, root and clazz might not be set yet.
			if (root == null) {
				root = superDesc.getRoot();
			}

			if (clazz == null) {
				clazz = superDesc.clazz;
			}

			inheritPropertyDescriptors(properties, superDesc.properties);
			inheritControlTasks(controlTasks, superDesc.controlTasks);
			inheritValidators(validators, superDesc.validators);

			for (Type superclass : superDesc.superclasses) {
				addSuperClass(superclass);
			}
			for (Type intf : superDesc.interfaces) {
				addInterface(intf);
			}
			toInitFrom = superDesc.superclasses.isEmpty() || superDesc.hierarchyInitialized ? null : superDesc.getSuperClasses().get(0);

            if (deployableType == null) {
                deployableType = superDesc.getDeployableType();
            }

            if (containerType == null) {
                containerType = superDesc.getContainerType();
            }
		} while (toInitFrom != null);

		syntheticPropertiesField = ReflectionUtils.searchField(clazz, ConfigurationItem.SYNTHETIC_PROPERTIES_FIELD);
		syntheticPropertiesField.setAccessible(true);

		verify();

		hierarchyInitialized = true;
	}

	private void verify() {
		verifySyntheticPropertiesField();
	    verifyReferenceTypes();
	    verifyNonArtifactDoesNotHavePlaceholders();
	    verifyArtifactInterfaces();
	    if(!isVirtual()) {
	    	verifyJavaClassIsNotAbstract();
	    	verifyDeployedHasDeployableAndContainerType();
			verifyHiddenRequiredPropertiesHaveDefaultValue();
		    verifyControlTasks();
		}
    }

	private void verifySyntheticPropertiesField() {
		checkNotNull(syntheticPropertiesField, "Synthetic properties field should be set");
		checkState(syntheticPropertiesField.getType().isAssignableFrom(Map.class), "Synthetic properties field should be Map<String, Object>, not: %s", syntheticPropertiesField.getType());
	}

	private void verifyControlTasks() {
		for (MethodDescriptor controlTask : controlTasks.values()) {
			controlTask.verify();
		}
	}

	private void verifyReferenceTypes() {
	    for(PropertyDescriptor p : properties.values()) {
	    	final PropertyKind kind = p.getKind();
			if(kind == CI || kind == SET_OF_CI || kind == LIST_OF_CI) {
	    		checkState(isValidReferencedType(p.getReferencedType()), "Property %s of type %s refers to non-existing type %s", p.getName(), p.getDeclaringDescriptor().getType(), p.getReferencedType());
	    	}
	    }
    }

	private boolean isValidReferencedType(Type referencedType) {
	    if(DescriptorRegistry.exists(referencedType))
	    	return true;

	    for(Descriptor d : DescriptorRegistry.getDescriptors()) {
	    	if(d.getSuperClasses().contains(referencedType)) {
	    		return true;
	    	}
	    	if(d.getInterfaces().contains(referencedType)) {
	    		return true;
	    	}
	    }

	    return false;
    }

	private void verifyNonArtifactDoesNotHavePlaceholders() {
		if (!isAssignableTo(Artifact.class)) {
			for (PropertyDescriptor propertyDescriptor : properties.values()) {
				checkState(!propertyDescriptor.getName().equals(PLACEHOLDERS_FIELD), "Type %s that does not implemnt the udm.Artifact interface must not have a field called 'placeholders'", type);
			}
		}
	}

	private void verifyArtifactInterfaces() {
		if(isAssignableTo(Deployable.class) && isAssignableTo(Artifact.class)) {
			checkState(isAssignableTo(SourceArtifact.class), "Type %s that implements the udm.Deployable and udm.Artifact interface must also implement the udm.SourceArtifact interface", type);
			checkState(isAssignableTo(DeployableArtifact.class), "Type %s that implements the udm.Deployable and udm.Artifact interface must also implement the udm.DeployableArtifact interface", type);
		}
		if(isAssignableTo(Deployed.class) && isAssignableTo(Artifact.class)) {
			checkState(isAssignableTo(DerivedArtifact.class), "Type %s that implements the udm.Deployed and udm.Artifact interface must also implement the udm.DerivedArtifact interface", type);
		}
    }

	private void verifyJavaClassIsNotAbstract() {
		checkState(!isAbstract(clazz.getModifiers()), "Non-virtual type %s has an abstract Java class %s", getType(), clazz.getName());
    }

	private void verifyDeployedHasDeployableAndContainerType() {
		Type deployedType = Type.valueOf(Deployed.class);
		if(isAssignableTo(deployedType)) {
			checkState(getDeployableType() != null, "Non-virtual type %s is a sub-type of %s but does not have a deployable-type", getType(), deployedType);
			checkState(getContainerType() != null, "Non-virtual type %s is a sub-type of %s but does not have a container-type", getType(), deployedType);
		}
    }

	private void verifyHiddenRequiredPropertiesHaveDefaultValue() {
	    for(PropertyDescriptor p : properties.values()) {
	    	if(p.isHidden() && p.isRequired()) {
	    		checkState(p.getDefaultValue() != null, "Hidden required property %s of non-virtual type %s must have a default value", p.getName(), getType());
	    	}
	    }
    }

	private void inheritValidators(List<Validator<ConfigurationItem>> dest, List<Validator<ConfigurationItem>> source) {
		dest.addAll(source);
	}

	private void inheritControlTasks(Map<String, MethodDescriptor> dest, Map<String, MethodDescriptor> source) {
		for (Map.Entry<String, MethodDescriptor> sourceEntry : source.entrySet()) {
			if (!dest.containsKey(sourceEntry.getKey())) {
				dest.put(sourceEntry.getKey(), new MethodDescriptor(sourceEntry.getValue(), this));
			} else {
				logger.warn("Not inheriting ControlTask [{}] on [{}]", new Object[] {sourceEntry.getValue().getFqn(), type});
			}
		}
	}

    private void inheritPropertyDescriptors(Map<String, PropertyDescriptor> dest, Map<String, PropertyDescriptor> source) {
        Map<String, PropertyDescriptor> myDescriptors = newLinkedHashMap(dest);
        dest.clear();
        for (Map.Entry<String, PropertyDescriptor> sourceEntry : source.entrySet()) {
            if (!myDescriptors.containsKey(sourceEntry.getKey())) {
                dest.put(sourceEntry.getKey(), new PropertyDescriptor(sourceEntry.getValue(), this));
            } else {
                myDescriptors.get(sourceEntry.getKey()).overrideWith(sourceEntry.getValue());
            }
        }
        dest.putAll(myDescriptors);
    }

    private void addSuperClass(Type supertype) {
		superclasses.add(supertype);
		DescriptorRegistry.registerSubtype(supertype, type);
	}

	private void addInterface(Type intf) {
		DescriptorRegistry.registerSubtype(intf, type);
		interfaces.add(intf);
	}

	private void addProperty(PropertyDescriptor propertyDescriptor) {
		properties.put(propertyDescriptor.getName(), propertyDescriptor);
	}

	private void addControlTask(MethodDescriptor from) {
		controlTasks.put(from.getName(), from);
	}

	public Type getType() {
		return type;
	}

	public Class<?> getClazz() {
		return clazz;
	}

	public String getDescription() {
		return description;
	}

	public Metadata.ConfigurationItemRoot getRoot() {
		return root;
	}

	public Collection<PropertyDescriptor> getPropertyDescriptors() {
		return properties.values();
	}

	public PropertyDescriptor getPropertyDescriptor(String name) {
		return properties.get(name);
	}

	public MethodDescriptor getControlTask(String name) {
		return controlTasks.get(name);
	}

	public Collection<MethodDescriptor> getControlTasks() {
		return controlTasks.values();
	}

	public boolean isAssignableTo(Class<?> clazz) {
		return isAssignableTo(Type.valueOf(clazz));
	}

	public boolean isAssignableTo(Type type) {
		return this.type.isSubTypeOf(type) || this.type.equals(type);
	}

	public List<Type> getSuperClasses() {
		return superclasses;
	}

	public Set<Type> getInterfaces() {
		return interfaces;
	}

	public boolean isVirtual() {
		return virtual;
	}

    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
    	return areEqual(item, other, new HashSet<String>());
    }

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        if (item == null) {
            return other == null;
        }

        if (!getType().equals(item.getType()) || !getType().equals(other.getType())) {
            return false;
        }

        if(!itemsBeingCompared.add(item.getId())) {
        	return true;
        }

        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            if (!pd.areEqual(item, other, itemsBeingCompared)) {
                return false;
            }
        }
        return true;
    }

    boolean shouldGenerateDeployableType() {
		return generatedDeployableType != null;
	}

	@SuppressWarnings("unchecked")
	public <T extends ConfigurationItem> T newInstance() {
		if (virtual) {
			throw new IllegalArgumentException("Cannot instantiate class for " + type + " because it is virtual");
		}

		try {
			Field typeField = searchField(clazz, ConfigurationItem.TYPE_FIELD);
			typeField.setAccessible(true);
			T t = (T) clazz.newInstance();
			typeField.set(t, type);

			prefillDefaultProperties(t);

			return t;
		} catch (InstantiationException exc) {
			throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
		} catch (IllegalAccessException exc) {
			throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
		}
	}

	private <T extends ConfigurationItem> void prefillDefaultProperties(T t) {
		for (PropertyDescriptor pd : getPropertyDescriptors()) {
			if (pd.getDefaultValue() != null) {
				pd.set(t, pd.getDefaultValue());
			} else {
				pd.set(t, pd.emptyValue());
			}
		}
	}

	@Override
	public String toString() {
		return "Descriptor[" + type + "]";
	}

	public Type getDeployableType() {
		return deployableType;
	}

	public Type getContainerType() {
		return containerType;
	}

	Field getSyntheticPropertiesField() {
		return syntheticPropertiesField;
	}

	public List<ValidationMessage> validate(final ConfigurationItem ci) {
		final ArrayList<ValidationMessage> messages = newArrayList();
		for (PropertyDescriptor propertyDescriptor : properties.values()) {
			propertyDescriptor.validate(ci, messages);
		}
		for (Validator<ConfigurationItem> validator : validators) {
			validator.validate(ci, new ValidationContext() {
				@Override
				public void error(String message, Object... params) {
					messages.add(new ValidationMessage(ci.getId(), null, String.format(message, params)));
				}
			});
		}
		return messages;
	}

	@SuppressWarnings("serial")
	private static class DescriptorException extends RuntimeException {
		public DescriptorException(String s, RuntimeException e) {
			super(s, e);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(Descriptor.class);
}
