package com.xebialabs.deployit.plugin.api.reflect;

import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.plugin.api.reflect.ReflectionUtils.handleInvocationTargetException;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getOptionalStringAttribute;
import static com.xebialabs.deployit.plugin.api.reflect.SyntheticHelper.getRequiredStringAttribute;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

import org.w3c.dom.Element;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;

public class MethodDescriptor {

	private static final String CONTROL_TASK_DISPATCH_METHOD = "controlTaskDispatch";

	private String name;
	private Method method;
	private String description;
	private Descriptor descriptor;

	private MethodDescriptor(Descriptor descriptor, String name) {
		this.descriptor = descriptor;
		this.name = name;
	}

	private MethodDescriptor(Descriptor descriptor, Method method) {
		this.method = method;
		this.name = method.getName();
		this.descriptor = descriptor;
	}

	MethodDescriptor(MethodDescriptor copyOf, Descriptor newOwner) {
		this.method = copyOf.method;
		this.name = copyOf.name;
		this.description = copyOf.description;
		this.descriptor = newOwner;
	}

	static MethodDescriptor from(Descriptor descriptor, Method method) {
		MethodDescriptor methodDescriptor = new MethodDescriptor(descriptor, method);
		methodDescriptor.initMetadata();
		return methodDescriptor;
	}

	static MethodDescriptor from(Descriptor descriptor, Element element) {
		String name = getRequiredStringAttribute(element, "name");
		MethodDescriptor methodDescriptor = new MethodDescriptor(descriptor, name);
		methodDescriptor.description = getOptionalStringAttribute(element, "description", "No description.");
		return methodDescriptor;
	}

	private void initMetadata() {
		ControlTask annotation = method.getAnnotation(ControlTask.class);
		description = annotation.description();
	}

	void verify() {
		if (method != null) {
			checkState(method.getParameterTypes().length == 0, "ControlTask %s should not take any parameters", getFqn());
			checkState(List.class.isAssignableFrom(method.getReturnType()), "ControlTask %s should return a List<Step>", getFqn());
		} else {
			Method controlTaskDispatch = getDispatcher();
			checkState(controlTaskDispatch.getParameterTypes().length == 1, "ControlTask dispatcher %s for %s should take name parameter", CONTROL_TASK_DISPATCH_METHOD, getFqn());
			checkState(List.class.isAssignableFrom(controlTaskDispatch.getReturnType()), "ControlTask dispatcher %s for %s should return a List<Step>", CONTROL_TASK_DISPATCH_METHOD, getFqn());
		}
	}

	public String getName() {
		return name;
	}

	public Method getMethod() {
		return method;
	}

	public String getDescription() {
		return description;
	}

	public Descriptor getDeclaringDescriptor() {
		return descriptor;
	}
	
	@SuppressWarnings("unchecked")
    public <T> T invoke(ConfigurationItem item) {
		try {
			if (method == null) {
				return (T) getDispatcher().invoke(item, name);
			} else {
				return (T) method.invoke(item);
			}
		} catch (IllegalAccessException e) {
			throw new RuntimeException("Could not invoke " + name + " on " + item, e);
		} catch (InvocationTargetException e) {
			throw handleInvocationTargetException(e, "Could not invoke " + name + " on " + item);
		}
	}

	public String getFqn() {
		return String.format("%s.%s", descriptor.getType(), name);
	}

	private Method getDispatcher() {
		Class<?> clazz = descriptor.getClazz();
		try {
			return clazz.getMethod(CONTROL_TASK_DISPATCH_METHOD, String.class);
		} catch (NoSuchMethodException e) {
			throw new IllegalStateException("No ControlTask dispatcher found for " + name + " on " + descriptor, e);
		}
	}
}
