package com.xebialabs.deployit.plugin.api.reflect;

import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.udm.Prefix;

import java.io.Serializable;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * The type of a {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}
 */
@SuppressWarnings("serial")
public class Type implements Serializable {

    private String prefix;
    private String name;

    private Type(String prefix, String name) {
        checkNotNull(prefix, "prefix");
        checkNotNull(name, "type");
        this.prefix = prefix;
        this.name = name;
    }

    public static Type valueOf(Class<?> ciClass) {
        checkNotNull(ciClass, "Cannot get the type of a null class");
        Package ciPackage = ciClass.getPackage();
        Prefix prefix = ciPackage.getAnnotation(Prefix.class);
        checkNotNull(prefix, "Package [%s] should have an @Prefix annotation for ci-class [%s]", ciPackage.getName(), ciClass.getName());
	    String simpleName = ciClass.getSimpleName();
	    checkNotNull(Strings.emptyToNull(simpleName), "Could not get a typename for ci-class [%s]", ciClass.getName());
	    return valueOf(prefix.value(), simpleName);
    }

    public static Type valueOf(String typeName) {
        checkNotNull(typeName, "Cannot get the type of a null typeName String");
        checkArgument(typeName.indexOf('.') != -1, "Type %s does not contain a prefix", typeName);
        int indexOfLastDot = typeName.lastIndexOf('.');
        return valueOf(typeName.substring(0, indexOfLastDot), typeName.substring(indexOfLastDot + 1));
    }

    public static Type valueOf(String prefix, String simpleName) {
        return new Type(prefix, simpleName);
    }

	public Descriptor getDescriptor() {
		return DescriptorRegistry.getDescriptor(this);
	}

	public boolean isSubTypeOf(Type superType) {
		return DescriptorRegistry.getSubtypes(superType).contains(this);
	}

	public boolean isSuperTypeOf(Type subType) {
		return DescriptorRegistry.getSubtypes(this).contains(subType);
	}

    public String getPrefix() {
        return prefix;
    }

    public String getName() {
        return name;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Type type1 = (Type) o;

        return prefix.equals(type1.prefix) && name.equals(type1.name);
    }

    @Override
    public int hashCode() {
        int result = prefix != null ? prefix.hashCode() : 0;
        result = 31 * result + (name != null ? name.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return prefix + "." + name;
    }
}
