/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.semver;

import com.github.zafarkhaja.semver.UnexpectedCharacterException;
import com.github.zafarkhaja.semver.Version;
import com.github.zafarkhaja.semver.util.Stream;
import com.github.zafarkhaja.semver.util.UnexpectedElementException;

import java.util.function.Predicate;

import static com.xebialabs.deployit.plugin.api.semver.Lexer.Token.Type.*;

public class VersionRange {

    private boolean rightClosed;
    private Version leftVersion;
    private Version rightVersion;
    private boolean leftClosed;

    public static VersionRange withDefaultLexer(final String range) {
        return new VersionRange(range);
    }

    public VersionRange(final String range) {
        this(new Lexer(), range);
    }

    public VersionRange(final Lexer lexer, final String range) {
        Stream<Lexer.Token> tokens = lexer.tokenize(range);
        leftClosed = tokens.positiveLookahead(LEFT_CLOSED);
        boolean leftOpen = tokens.positiveLookahead(LEFT_OPEN);
        try {
            if (!leftClosed && !leftOpen) {
                parseSingleVersionRange(range);
            } else {
                    tokens.consume(leftClosed ? LEFT_CLOSED : LEFT_OPEN);
                    leftVersion = extractVersion(tokens, tokenStream -> !tokenStream.positiveLookahead(COMMA));
                    tokens.consume(COMMA);
                    rightVersion = extractVersion(tokens, tokenStream -> !tokenStream.positiveLookahead(RIGHT_CLOSED, RIGHT_OPEN));
                    rightClosed = tokens.positiveLookahead(RIGHT_CLOSED);
                    tokens.consume(rightClosed ? RIGHT_CLOSED : RIGHT_OPEN);
            }
        } catch (NumberFormatException | UnexpectedElementException | UnexpectedCharacterException e) {
            throw createIllegalArgumentException(range, e);
        }
    }

    private void parseSingleVersionRange(String range) {
        leftClosed = true;
        rightClosed = true;
        leftVersion = parseVersion(range);
        rightVersion = leftVersion;
    }

    private IllegalArgumentException createIllegalArgumentException(String range, RuntimeException e) {
        return new IllegalArgumentException(String.format("Invalid range: '%s'.", range), e);
    }

    private Version extractVersion(Stream<Lexer.Token> tokens, Predicate<Stream<Lexer.Token>> predicate) {
        StringBuilder versionBuilder = new StringBuilder();
        while (tokens.lookahead() != null && predicate.test(tokens)) {
            versionBuilder.append(tokens.consume().getLexeme());
        }
        return parseVersion(versionBuilder.toString());
    }

    private Version parseVersion(final String version) {
        Version parsedVersion = null;
        String trimmedVersion = version.trim();
        if(!trimmedVersion.isEmpty()) {
            String[] parts = trimmedVersion.split("\\.");
            if (parts.length == 1) {
                parsedVersion = Version.forIntegers(Integer.parseInt(parts[0]));
            } else if (parts.length == 2) {
                parsedVersion = Version.forIntegers(Integer.parseInt(parts[0]), Integer.parseInt(parts[1]));
            } else {
                parsedVersion = Version.valueOf(version);
            }
        }
        return parsedVersion;
    }

    public boolean includes(Version version) {
        return (leftVersion == null || leftVersion.compareTo(version) < (leftClosed ? 1 : 0)) &&
                (rightVersion == null || rightVersion.compareTo(version) >= (rightClosed ? 0 : 1));
    }


}