/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.validation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Collection;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;

/**
 * The annotated element size must be between the specified boundaries (included).
 * <p>
 * Supported types are:
 * <ul>
 *      <li>{@code STRING} (length of string is evaluated)</li>
 *      <li>{@code SET_OF_STRING} (set size is evaluated)</li>
 *      <li>{@code SET_OF_CI} (set size is evaluated)</li>
 *      <li>{@code LIST_OF_STRING} (list size is evaluated)</li>
 *      <li>{@code LIST_OF_CI} (list size is evaluated)</li>
 *      <li>{@code MAP_STRING_STRING} (map size is evaluated)</li>
 * </ul>
 * </p>
 * {@code null} elements are considered valid.
 */
@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = Size.Validator.class, type = "size")
@ApplicableTo({PropertyKind.STRING, PropertyKind.SET_OF_STRING, PropertyKind.SET_OF_CI, PropertyKind.LIST_OF_STRING, PropertyKind.LIST_OF_CI, PropertyKind.MAP_STRING_STRING})
@Target(ElementType.FIELD)
public @interface Size {
    String DEFAULT_MESSAGE = "The size must be between %s and %s";
    int MIN_VALUE = 0;
    int MAX_VALUE = Integer.MAX_VALUE;

    /**
     * @return size the element must be higher or equal to
     */
    int min() default MIN_VALUE;

    /**
     * @return size the element must be lower or equal to
     */
    int max() default MAX_VALUE;

    String message() default DEFAULT_MESSAGE;

    class Validator implements com.xebialabs.deployit.plugin.api.validation.Validator<Object> {
        private int min = Size.MIN_VALUE;

        private int max = Size.MAX_VALUE;

        private String message = Size.DEFAULT_MESSAGE;

        @Override
        public void validate(final Object value, final ValidationContext context) {
            if (min < 0) {
                context.error("The min parameter cannot be negative.");
            }
            if (max < 0) {
                context.error("The max parameter cannot be negative.");
            }
            if (max < min) {
                context.error("The length cannot be negative.");
            }
            if (value != null) {
                if (value instanceof String) {
                    int length = ((String) value).length();
                    if (isNotValid(length)) {
                        context.error(message, min, max);
                    }
                } else if (value instanceof Collection) {
                    int size = ((Collection<?>) value).size();
                    if (isNotValid(size)) {
                        context.error(message, min, max);
                    }
                } else if (value instanceof Map) {
                    int size = ((Map<?, ?>) value).size();
                    if (isNotValid(size)) {
                        context.error(message, min, max);
                    }
                }
            }
        }

        private boolean isNotValid(int length) {
            return length < min || length > max;
        }

        public int getMin() {
            return min;
        }

        public int getMax() {
            return max;
        }

        public String getMessage() {
            return message;
        }
    }
}
