/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.validation.ExtendedValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * Encapsulates the metadata about a type in the XL Deploy type system. Example of such metadata:
 * the available properties, the implemented interfaces, the available control tasks etc.
 * <p>
 * The Descriptor for a specific type can be retrieved using the MetadataService,
 * which is also available as a REST API.
 */
@PublicApiRef
public interface
Descriptor {

    /**
     * @return The type of a {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     */
    Type getType();

    /**
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} class object.
     */
    Class<?> getClazz();

    /**
     * @return A textual description of this type.
     */
    String getDescription();

    /**
     * @return The friendly name of the type
     */
    String getLabel();

    /**
     * @return The root in the JCR tree.
     */
    default Metadata.ConfigurationItemRoot getRoot() {
        if (getRootName() == null || getRootName().isEmpty()) {
            return Metadata.ConfigurationItemRoot.NESTED;
        }
        for (Metadata.ConfigurationItemRoot root : Metadata.ConfigurationItemRoot.values()) {
            if (getRootName().equalsIgnoreCase(root.getRootNodeName())) {
                return root;
            }
        }
        return Metadata.ConfigurationItemRoot.BY_ROOT_NAME;
    }

    /**
     * @return The root name in the JCR tree.
     */
    String getRootName();

    /**
     * Note: The return-type is a Collection, but a List is actually returned. So the order of the properties is guaranteed.
     * We cannot change the return-type as this would break binary compatibility.
     *
     * @return A list of metadata about a all the properties of this type.
     */
    Collection<PropertyDescriptor> getPropertyDescriptors();

    /**
     * @param name The name of the property.
     * @return The metadata for the given property name.
     */
    PropertyDescriptor getPropertyDescriptor(String name);

    /**
     * @param name The name of the control methods.
     * @return The metadata for the given a control method name.
     */
    MethodDescriptor getControlTask(String name);

    /**
     * @return A list of metadata about a all the control methods of this type.
     */
    Collection<MethodDescriptor> getControlTasks();

    /**
     * @param clazz A {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} class object.
     * @return Whether this type is a subtype or has the same type as the type of the given CI class.
     */
    boolean isAssignableTo(Class<?> clazz);

    /**
     * @param type A CI type.
     * @return Whether this type is a subtype or has the same type as the given type.
     */
    boolean isAssignableTo(Type type);

    /**
     * @return The list of all the super classes of this {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     */
    List<Type> getSuperClasses();

    /**
     * @return The set of all the interfaces this {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} implements.
     */
    Set<Type> getInterfaces();

    /**
     * @return Whether this type is a virtual type.
     */
    boolean isVirtual();

    /**
     * @return Whether or not this is a versioned type, i.e. whether or not the JCR version history for CIs of this type is kept.
     */
    boolean isVersioned();

    /**
     * @return The relative path to the icon for this type.
     */
    String getIcon();

    /**
     * @return Whether or not the given CIs are considered to be equal.
     */
    boolean areEqual(ConfigurationItem item, ConfigurationItem other);

    /**
     * @param id  The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} id.
     * @param <T> The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} class.
     * @return A new CI instance of the given type and id.
     */
    <T extends ConfigurationItem> T newInstance(String id);

    @Override
    String toString();

    /**
     * Only applicable for {@link com.xebialabs.deployit.plugin.api.udm.Deployed}s.
     *
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Deployable} type that created this deployed.
     */
    Type getDeployableType();

    /**
     * Only applicable for {@link com.xebialabs.deployit.plugin.api.udm.Deployed}s.
     *
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Container} type this deployed can be deployed to.
     */
    Type getContainerType();

    /**
     * Validates the values for each property of the CI.
     *
     * @param ci The {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} to ve validated.
     * @return A list of {@link ValidationMessage}s.
     */
    List<ValidationMessage> validate(ConfigurationItem ci);

    List<ValidationMessage> validate(ExtendedValidationContext context, ConfigurationItem ci);

    List<ValidationMessage> validateInputHint(ConfigurationItem ci);

    /**
     * @return Whether this type supports inspection.
     */
    boolean isInspectable();

    CreatorDescriptor getCreator();

}
