/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;

import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.Map;
import java.util.Objects;

/**
 * Provide methods to get a value by key from a storage.
 */
@PublicApi
public interface IDictionary extends ConfigurationItem {

    /**
     * @param key The key of a dictionary entry.
     * @return The value associated with the given key.
     */
    String getValue(String key);

    /**
     * @return The dictionary entries.
     */
    Map<String, String> getEntries();

    /**
     * Return a view on this dictionary as applicable to the given context.
     *
     * @param context The context to generate the view for.
     * @return a dictionary tailored for the given context. Can be <code>this</code> if no adjustments are required or <code>null</code> if the dictionary is not applicable to the given context.
     */
    IDictionary applyTo(DictionaryContext context);

    /**
     * Context used when processing dictionaries for a deployment.
     */
    class DictionaryContext {

        private final Application application;

        private final Version applicationVersion;

        private final Environment environment;

        private final Container container;

        public DictionaryContext(Application application, Version applicationVersion, Environment environment, Container container) {
            this.application = application;
            this.applicationVersion = applicationVersion;
            this.environment = environment;
            this.container = container;
        }

        /**
         * @return The {@link Application} being deployed. Can be <code>null</code> when (effective) dictionary is processed without application.
         */
        public Application getApplication() {
            return application;
        }

        /**
         * @return The {@link Version} being deployed. Can be <code>null</code> when (effective) dictionary is processed without application.
         */
        public Version getApplicationVersion() {
            return applicationVersion;
        }

        /**
         * @return The {@link Environment} deploying to.
         */
        public Environment getEnvironment() {
            return environment;
        }

        /**
         * @return The {@link Container} the dictionary is going to be used for. Optional: Can be <code>null</code> when the dictionary is applied to a larger context.
         */
        public Container getContainer() {
            return container;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) return true;
            if (!(o instanceof DictionaryContext)) return false;
            final DictionaryContext that = (DictionaryContext) o;
            return Objects.equals(getApplication(), that.getApplication()) &&
                    Objects.equals(getApplicationVersion(), that.getApplicationVersion()) &&
                    Objects.equals(getEnvironment(), that.getEnvironment()) &&
                    Objects.equals(getContainer(), that.getContainer());
        }

        @Override
        public int hashCode() {
            return Objects.hash(application, applicationVersion, environment, container);
        }
    }
}
