/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;

import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.lang.annotation.*;

import static com.xebialabs.deployit.plugin.api.udm.Property.Size.DEFAULT;

/**
 * Specifies that this field is part of the configuration item. It will be injected with the correct value to be read from runbooks.
 * <p>
 * <b>N.B.:</b> If this annotation is used on a field of an {@link Enum} type, the enum should <b>not</b> have a toString method defined.
 */
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.FIELD)
public @interface Property {

    /**
     * Whether this property is modeled as a parent/child containment instead of a foreign key reference in the JCR tree.
     */
    boolean asContainment() default false;

    /**
     * Whether this property is a nested CI.
     */
    boolean nested() default false;

    /**
     * The category of this property. In the user interface all the properties belonging to the same category will be grouped together.
     */
    String category() default "Common";

    /**
     * The descriptive id of this property as it should be shown in the user interface. If no id is not specified, the name of the field is used. Camel
     * case in the field name is replaced by spaces and the first character is capitalized, e.g. "classLoaderPolicy" is transformed into "Class loader policy".
     */
    String label() default "";

    /**
     * The description of this property as it should be shown in the user interface. If no description is specified, the id is used.
     */
    String description() default "";

    /**
     * Whether this property is required.
     */
    boolean required() default true;

    /**
     * Whether this property is a password property. If <tt>true</tt> this field will be masked in the user interface.
     */
    boolean password() default false;

    /**
     * The size of this property as it should be shown in the user interface.
     */
    Size size() default DEFAULT;

    /**
     * A Default value for a Property. Can only be a String due to Java limitations. PropertyDescriptor will convert it for you.
     */
    String defaultValue() default "";

    /**
     * Whether this property is hidden, ie. cannot be changed via a client.
     */
    boolean hidden() default false;

    /**
     * Whether or not this property is transient, ie. it won't be persisted when the CI is persisted.
     * N.B.: You cannot use 'real' transient fields, due to the recovery of tasks.
     */
    boolean isTransient() default false;

    /**
     * The {@link CandidateValuesFilter} to apply when selecting values for this CI, SET_OF_CI, LIST_OF_CI property
     * @return The name of the {@link CandidateValuesFilter}
     */
    String candidateValuesFilter() default "";

    /**
     * Whether or not this property is readonly. If readonly, the property cannot be updated.
     */
    boolean readonly() default false;

    /**
     * Whether to use getter and setter to access the property
     */
    boolean useGetterAndSetter() default false;

    /**
     * Defines a number of standard sizes for configuration item properties.
     */
    @PublicApiRef
    public static enum Size {
        /**
         * Defines that the configuration item property should be shown using the default size.
         */
        DEFAULT,

        /**
         * Defines that the configuration item property should be shown using the small size.
         */
        SMALL,

        /**
         * Defines that the configuration item property should be shown using the medium size.
         */
        MEDIUM,

        /**
         * Defines that the configuration item property should be shown using the large size.
         */
        LARGE
    }
}
