/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.validation;

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Collection;
import java.util.regex.Pattern;

@Retention(RetentionPolicy.RUNTIME)
@Rule(clazz = Regex.Validator.class, type = "regex")
@ApplicableTo({PropertyKind.STRING, PropertyKind.LIST_OF_STRING, PropertyKind.SET_OF_STRING})
@Target(ElementType.FIELD)
public @interface Regex {
    String DEFAULT_MESSAGE = "Value '%s' did not conform to pattern %s";

    String pattern();

    String message() default DEFAULT_MESSAGE;

    public static class Validator implements PatternValidator<Object> {
        private String pattern;
        private String message = DEFAULT_MESSAGE;

        @Override
        @SuppressWarnings({"unchecked"})
        public void validate(Object value, ValidationContext context) {
            if (value instanceof String) {
                validateString((String) value, context);
            }
            if (value instanceof Collection<?>) {
                Collection<String> c = (Collection<String>) value;
                for (String s : c) {
                    validateString(s, context);
                }
            }
        }

        private void validateString(String s, ValidationContext context) {
            if (!Pattern.compile(pattern).matcher(s).matches()) {
                context.error(message, s, pattern);
            }
        }

        public String getPattern() {
            return pattern;
        }

        public void setPattern(String pattern) {
            this.pattern = pattern;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }
}