/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.tomcat.runbook;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.mapper.artifact.*;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatServer;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatUnmanagedServer;
import com.xebialabs.deployit.plugin.tomcat.mapper.*;
import com.xebialabs.deployit.plugin.tomcat.step.TomcatStartServerStep;
import com.xebialabs.deployit.plugin.tomcat.step.TomcatStopServerStep;
import com.xebialabs.deployit.plugin.tomcat.step.TomcatWaitStep;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Supports the deployment, re-deployment and undeployment of an
 * {@link com.xebialabs.deployit.ci.DeploymentPackage Application Package} to a
 * {@link TomcatUnmanagedServer Tomcat Servlet Container}.
 * <p/>
 * <h4>Conditions</h4>
 * <p/>
 * Will trigger if the change plan contains the addition, modification or
 * deletion of a {@link Deployment Deployment} CI to an
 * {@link com.xebialabs.deployit.ci.Environment Environment} CI which contains
 * Tomcat middleware CI's.
 * <p/>
 * Note that Additions, Modifications and Deletions are always handled in one
 * single flow.
 * <p/>
 * <h4>Actions</h4>
 * <p/>
 * <ol>
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.War war} from
 * {@link com.xebialabs.deployit.plugin.tomcat.ci.TomcatUnmanagedServer Tomcat
 * Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.War war} from
 * {@link com.xebialabs.deployit.plugin.tomcat.ci.TomcatUnmanagedServer Tomcat
 * Server Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.Libraries} from
 * {@link com.xebialabs.deployit.ci.Host}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles}
 * from {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.Libraries} on
 * {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} on
 * {@link com.xebialabs.deployit.ci.Host}
 * </ol>
 */
public class TomcatDeploymentRunBook extends SingleTypeHandlingRunBook<Deployment> {

	public TomcatDeploymentRunBook() {
		super(Deployment.class);
	}

	@SuppressWarnings("unchecked")
	@Override
	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		WarToTomcatUnmanagedServerMapper warToServerMapper = new WarToTomcatUnmanagedServerMapper(change);
		WarToTomcatManagedServerMapper warToServerWithManagerMapper = new WarToTomcatManagedServerMapper(change);
		DatasourceToTomcatServerMapper datasourceMapper = new DatasourceToTomcatServerMapper(change, warToServerMapper);
		ActiveMQConnectionFactoryToTomcatServerMapper amqcfMapper = new ActiveMQConnectionFactoryToTomcatServerMapper(change, warToServerMapper);
		TomcatJndiResourcesToTomcatServerMapper jndiMapper = new TomcatJndiResourcesToTomcatServerMapper(change,warToServerMapper);
		LibrariesToHostMapper librariesMapper = new LibrariesToHostMapper(change);
		LibrariesToTomcatServerMapper librariesToTomcatServerMapper = new LibrariesToTomcatServerMapper(change);
		GenericFolderToHostMapper genericFolderToHostMapper = new GenericFolderToHostMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);
		ConfigurationFilesToTomcatServerMapper configurationFilesToTomcatMapper = new ConfigurationFilesToTomcatServerMapper(change);
		ContextFilesToTomcatServerMapper contextFilesToTomcatServerMapper = new ContextFilesToTomcatServerMapper(change);
		SqlFolderToDatabaseMapper sqlFoldermapper = new SqlFolderToDatabaseMapper(change);
		SqlScriptToDatabaseMapper sqlScriptMapper = new SqlScriptToDatabaseMapper(change);


		Set<TomcatUnmanagedServer> affectedServers = new HashSet<TomcatUnmanagedServer>();
		affectedServers.addAll(warToServerMapper.getAffectedTargets());

		if (!isTomcatDeployment(warToServerMapper, warToServerWithManagerMapper, datasourceMapper)) {
			return;
		}

		stopServers(affectedServers, steps);


		datasourceMapper.generateDeletionSteps(steps);
		amqcfMapper.generateDeletionSteps(steps);
		jndiMapper.generateDeletionSteps(steps);

		warToServerMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateDeletionSteps(steps);
		configurationFilesToTomcatMapper.generateDeletionSteps(steps);
		contextFilesToTomcatServerMapper.generateDeletionSteps(steps);
		librariesMapper.generateDeletionSteps(steps);
		librariesToTomcatServerMapper.generateDeletionSteps(steps);
		genericFolderToHostMapper.generateDeletionSteps(steps);

		sqlFoldermapper.generateAdditionSteps(steps);
		sqlScriptMapper.generateAdditionSteps(steps);

		genericFolderToHostMapper.generateAdditionSteps(steps);
		librariesMapper.generateAdditionSteps(steps);
		librariesToTomcatServerMapper.generateAdditionSteps(steps);
		contextFilesToTomcatServerMapper.generateAdditionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);
		configurationFilesToTomcatMapper.generateAdditionSteps(steps);
		datasourceMapper.generateAdditionSteps(steps);
		amqcfMapper.generateAdditionSteps(steps);
		jndiMapper.generateAdditionSteps(steps);
		warToServerMapper.generateAdditionSteps(steps);

		startServers(affectedServers, steps);

		warToServerWithManagerMapper.generateDeletionSteps(steps);
		warToServerWithManagerMapper.generateAdditionSteps(steps);

	}

	private boolean isTomcatDeployment(WarToTomcatUnmanagedServerMapper warToServerMapper,
	                                   WarToTomcatManagedServerMapper warToServerWithManagerMapper, DatasourceToTomcatServerMapper datasourceMapper) {
		Set<TomcatServer> allServers = new HashSet<TomcatServer>();
		allServers.addAll(warToServerMapper.getAllTargets());
		allServers.addAll(warToServerWithManagerMapper.getAllTargets());
		allServers.addAll(datasourceMapper.getAllTargets());
		return !allServers.isEmpty();
	}

	private void stopServers(Set<TomcatUnmanagedServer> servers, List<Step> steps) {
		boolean toWait = false;
		for (TomcatUnmanagedServer eachServer : servers) {
			steps.add(new TomcatStopServerStep(eachServer));
			toWait = true;
		}

		if (toWait) {
			steps.add(new TomcatWaitStep());
		}
	}

	private void startServers(Set<TomcatUnmanagedServer> servers, List<Step> steps) {
		boolean toWait = false;
		for (TomcatUnmanagedServer eachServer : servers) {
			steps.add(new TomcatStartServerStep(eachServer));
			toWait = true;
		}

		if (toWait) {
			steps.add(new TomcatWaitStep());
		}
	}
}
