/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.tomcat.runbook;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatServer;
import com.xebialabs.deployit.plugin.tomcat.mapper.DatasourceToTomcatServerMapper;
import com.xebialabs.deployit.plugin.tomcat.mapper.WarToTomcatServerMapper;
import com.xebialabs.deployit.plugin.tomcat.mapper.WarToTomcatServerWithManagerMapper;
import com.xebialabs.deployit.plugin.tomcat.step.TomcatStartServerStep;
import com.xebialabs.deployit.plugin.tomcat.step.TomcatStopServerStep;
import com.xebialabs.deployit.plugin.tomcat.step.TomcatWaitStep;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

/**
 * Supports the deployment, re-deployment and undeployment of an
 * {@link com.xebialabs.deployit.ci.DeploymentPackage Application Package} to a
 * {@link TomcatServer Tomcat Servlet Container}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or
 * deletion of a {@link Deployment Deployment} CI to an
 * {@link com.xebialabs.deployit.ci.Environment Environment} CI which contains
 * Tomcat middleware CI's.
 * 
 * Note that Additions, Modifications and Deletions are always handled in one
 * single flow.
 * 
 * <h4>Actions</h4>
 * 
 * <ol>
 * <li>Undeploy
 * {@link com.xebialabs.deployit.ci.artifact.plugins.was.runbooks.War war} from
 * {@link com.xebialabs.deployit.plugin.tomcat.ci.TomcatServer Tomcat Server}
 * <li>Deploy
 * {@link com.xebialabs.deployit.ci.artifact.plugins.was.runbooks.War war} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss
 * Application Server}
 * </ol>
 * 
 */
public class TomcatDeploymentRunBook extends SingleTypeHandlingRunBook<Deployment> {

	public TomcatDeploymentRunBook() {
		super(Deployment.class);
	}

	@SuppressWarnings("unchecked")
	@Override
	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		WarToTomcatServerMapper warToServerMapper = new WarToTomcatServerMapper(change);
		WarToTomcatServerWithManagerMapper warToServerWithManagerMapper = new WarToTomcatServerWithManagerMapper(change);
		DatasourceToTomcatServerMapper datasourceMapper = new DatasourceToTomcatServerMapper(change, warToServerMapper);

		Set<TomcatServer> affectedServers = new HashSet<TomcatServer>();
		affectedServers.addAll(warToServerMapper.getAffectedTargets());

		checkForServerConfigurationErrors(affectedServers);
		
		if (!isTomcatDeployment(warToServerMapper)) {
			return;
		}

		stopServers(affectedServers, steps);
		datasourceMapper.generateDeletionSteps(steps);
		warToServerMapper.generateDeletionSteps(steps);
		datasourceMapper.generateAdditionSteps(steps);
		warToServerMapper.generateAdditionSteps(steps);
		startServers(affectedServers, steps);
		
		warToServerWithManagerMapper.generateDeletionSteps(steps);
		warToServerWithManagerMapper.generateAdditionSteps(steps);

	}

	private void checkForServerConfigurationErrors(Set<TomcatServer> affectedServers) {
		for(TomcatServer server : affectedServers) {
	       if(!server.isManagerAppAvailable() && !server.isStartStopCommandAvailable()) {
	    	   throw new ResolutionException("Server " + server.getLabel() + " has neither start/stop commands nor manager app info configured");
	       }
		}
	}

	private boolean isTomcatDeployment(WarToTomcatServerMapper warToServerMapper) {
		Set<TomcatServer> allServers = new HashSet<TomcatServer>();
		allServers.addAll(warToServerMapper.getAllTargets());
		return !allServers.isEmpty();
	}

	private void stopServers(Set<TomcatServer> servers, List<Step> steps) {
		boolean toWait = false;
		for (TomcatServer eachServer : servers) {
			if (!eachServer.isManagerAppAvailable()) {
				steps.add(new TomcatStopServerStep(eachServer));
				toWait = true;
			}
		}

		if (toWait) {
			steps.add(new TomcatWaitStep());
		}
	}

	private void startServers(Set<TomcatServer> servers, List<Step> steps) {
		for (TomcatServer eachServer : servers) {
			if (!eachServer.isManagerAppAvailable()) {
				steps.add(new TomcatStartServerStep(eachServer));
			}
		}
	}
}
