package com.xebialabs.deployit.plugin.tomcat.ci;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.ci.artifact.mapping.WarMapping;
import com.xebialabs.deployit.hostsession.HostSession;

/**
 * A Apache Tomcat server
 */
@SuppressWarnings("serial")
@ConfigurationItem(description = "Tomcat Server instance", targetForMappingTypes = { WarMapping.class })
public class TomcatServer extends BaseConfigurationItem {

	public static final int DEPLOYMENT_COMPLETION_WAIT_TIME_MILLIS = 3000;

	@ConfigurationItemProperty(required = true, label = "Tomcat server host", description = "Host on which the Tomcat server is running.")
	private Host host;

	@ConfigurationItemProperty(required = true, description = "Port for the Tomcat Server, default is 8080")
	private int port;

	@ConfigurationItemProperty(required = true, description = "Protocol to be used for accessing the tomcat server")
	private AccessMethod accessMethod;

	@ConfigurationItemProperty(required = true, label = "Tomcat installation location", description = "Place where Tomcat is installed such as /opt/apache-tomcat-6.0.24.", size = Size.LARGE)
	private String tomcatHome;

	@ConfigurationItemProperty(required = false, label = "Tomcat stop command", description = "Command that should be executed to stop the Tomcat server, e.g. /opt/tomat/bin/catalina.sh stop", size = Size.LARGE)
	private String stopCommand;

	@ConfigurationItemProperty(required = false, label = "Tomcat start command", description = "Command that should be executed to start the Tomcat server, e.g./opt/tomcat/bin/catalina.sh start.", size = Size.LARGE)
	private String startCommand;

	@ConfigurationItemProperty(required = false, label = "Manager application context", description = "relative context path for tomcat manager application, e.g. /manager", size = Size.DEFAULT)
	private String managerAppContext;

	@ConfigurationItemProperty(required = false, label = "Tomcat manager username", description = "Username to be used for the manager application deployed on tomcat", size = Size.DEFAULT)
	private String managerUsername;

	@ConfigurationItemProperty(required = false, label = "Tomcat manager password", description = "Password to be used for the manager application deployed on tomcat", size = Size.DEFAULT, password = true)
	private String managerPassword;
	
	@ConfigurationItemProperty(required = false, description = "absolute path of the directory where wars are deployed", size = Size.DEFAULT)
	private String appBase;

	public HostSession connectToAdminHost() {
		return getHost().getHostSession();
	}

	public String getDeployDirPath() {
		String fileSep = getHost().getFileSeparator();
		return getTomcatHome() + fileSep + "webapps";
	}

	public String getContextWithoutVirtualHostDirPath() {
		String fileSep = getHost().getFileSeparator();
		return getTomcatHome() + fileSep + "conf" + fileSep + "Catalina" + fileSep + "localhost";
	}

	public String getContextWithVirtualHostDirPath(String virtualHostName) {
		String fileSep = getHost().getFileSeparator();
		return getTomcatHome() + fileSep + "conf" + fileSep + "Catalina" + fileSep + virtualHostName;
	}
	

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public String getStopCommand() {
		return stopCommand;
	}

	public void setStopCommand(String stopCommand) {
		this.stopCommand = stopCommand;
	}

	public String getStartCommand() {
		StringBuffer command = new StringBuffer();
		
		if (getHost().getOperatingSystemFamily() == OperatingSystemFamily.UNIX) {
			command.append("nohup ");
		}
		if (!StringUtils.isEmpty(startCommand)) {
			command.append(startCommand);
		}
		
		return command.toString();
	}

	public boolean isReachable() {
		boolean reachable = false;
		try {
			URL tomcatHomePageUrl = new URL(accessMethod.getValue() + "://" + getHost().getAddress() + ":" + getPort());
			BufferedReader in = new BufferedReader(new InputStreamReader(tomcatHomePageUrl.openStream()));
			in.readLine();
			in.close();
			reachable = true;
		} catch (Exception e) {
			// do not do anything!!
		}
		return reachable;
	}

	public boolean isStartStopCommandAvailable() {
		return !StringUtils.isBlank(startCommand) && !StringUtils.isBlank(stopCommand);
	}

	public boolean isManagerAppAvailable() {
		return !StringUtils.isBlank(managerAppContext) && !StringUtils.isBlank(managerUsername)
				&& !StringUtils.isBlank(managerPassword);
	}

	public void setStartCommand(String startCommand) {
		this.startCommand = startCommand;
	}

	public String getTomcatHome() {
		return tomcatHome;
	}

	public void setTomcatHome(String tomcatHome) {
		this.tomcatHome = tomcatHome;
	}

	public int getPort() {
		return port != 0 ? port : 8080;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getManagerUsername() {
		return managerUsername;
	}

	public void setManagerUsername(String managerUsername) {
		this.managerUsername = managerUsername;
	}

	public String getManagerPassword() {
		return managerPassword;
	}

	public void setManagerPassword(String managerPassword) {
		this.managerPassword = managerPassword;
	}

	public AccessMethod getAccessMethod() {
		return accessMethod;
	}

	public void setAccessMethod(AccessMethod accessMethod) {
		this.accessMethod = accessMethod;
	}

	public String getManagerAppContext() {
		return managerAppContext;
	}

	public void setManagerAppContext(String managerAppContext) {
		this.managerAppContext = managerAppContext;
	}

	public String getAppBase() {
		return appBase;
	}

	public void setAppBase(String appBase) {
		this.appBase = appBase;
	}

}
