/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.test.support.onthefly;

import static com.xebialabs.deployit.test.support.utils.VelocityUtils.evaluateTemplate;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import org.apache.velocity.VelocityContext;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.ci.artifact.Ear;

/**
 * Constructs an EAR for testing, accepting multiple {@link WarOnTheFly WARs} as entries. 
 * The context root for a WAR in the EAR is its name. The actual EAR file is finally written 
 * to a temporary directory. Example:
 * 
 * <pre>
 * EarOnTheFly earOnTheFly = new EarOnTheFly("myEar");
 * earOnTheFly.addFile("foo", someFile)
 *            ...
 *            .addFile("bar", someResource);
 * WarOnTheFly warOnTheFly = new WarOnTheFly("myWar");
 * warOnTheFly.addServlet(FooServlet.class);
 * earOnTheFly.addWarOnTheFly(warOnTheFly); // will have context root http://myhost:myport/myWar
 * File ear = earOnTheFly.writeToTemporaryFile(); // something like myEarXX.ear
 * </pre>
 * 
 * See also &quot;<a href="http://blog.xebia.com/2009/12/14/middleware-integration-testing-with-junit-maven-and-vmware-part-2/">
 * Middleware integration testing with JUnit, Maven and VMware: part 2</a>&quot;.
 * 
 * @see WarOnTheFly
 * @see JarOnTheFly
 */
public class EarOnTheFly extends JarOnTheFly {
	protected static final String EAR_EXTENSION = '.' + Ear.ARCHIVE_EXTENSION;
	
	private final Map<String, String> wars = Maps.newHashMap();

	public EarOnTheFly(String name) {
		super(name);
	}

	public EarOnTheFly addWarOnTheFly(WarOnTheFly wotf) throws IOException {
		String warFilename = wotf.getName() + WarOnTheFly.WAR_EXTENSION;
		Resource warFile = new FileSystemResource(wotf.writeToTemporaryFile());
		addFile(warFilename, warFile);
		wars.put(wotf.getName(), warFilename);
		return this;
	}

	public File writeToTemporaryFile() throws IOException {
		addApplicationXml();
		return writeToTemporaryFile(name, EAR_EXTENSION);
	}

	private void addApplicationXml() {
		VelocityContext context = new VelocityContext();
		context.put("name", name);
		context.put("wars", wars);
		String webxml = evaluateTemplate(context, "com/xebialabs/deployit/test/support/onthefly/application.xml.vm");
		addFile("META-INF/application.xml", new ByteArrayResource(webxml.getBytes()));
	}

}
