/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.test.support.onthefly;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.jar.Attributes.Name;

import org.apache.commons.io.IOUtils;
import org.springframework.core.io.Resource;

import com.google.common.collect.Maps;

/**
 * Constructs a JAR for testing, accepting multiple {@link Resource resources} as entries
 * and writing the actual JAR file to a temporary directory. Example:
 * 
 * <pre>
 * JarOnTheFly jarOnTheFly = new JarOnTheFly("myJar");
 * jarOnTheFly.addFile("foo", someFile)
 *            ...
 *            .addFile("bar", someResource);
 * File jar = jarOnTheFly.writeToTemporaryFile(); // something like myJarXX.jar
 * </pre>
 * 
 * See also &quot;<a href="http://blog.xebia.com/2009/12/14/middleware-integration-testing-with-junit-maven-and-vmware-part-2/">
 * Middleware integration testing with JUnit, Maven and VMware: part 2</a>&quot;.
 */
public class JarOnTheFly {
	protected static final String JAR_EXTENSION = ".jar";
	
	protected final String name;

	private final Map<String, Resource> files = Maps.newHashMap();
	private final ManifestDefinition manifestDefinition = new ManifestDefinition();

	public JarOnTheFly(String name) {
		this.name = name;
	}

	protected void addManifestMainAttribute(String name, String value) {
		manifestDefinition.addMainAttribute(name, value);
	}
	
	// can't use "addFile" since there may not be a resource corresponding to the directory
	protected JarOnTheFly addManifestDirectoryAttributes(String dirname, Map<String, String> manifestAttributes) {
		manifestDefinition.addEntryAttributes(dirname, checkNotNull(manifestAttributes));
		return this;
	}

	public JarOnTheFly addFile(String filename, Resource resource) {
		return addFile(filename, resource, null);
	}
	
	protected JarOnTheFly addFile(String filename, Resource resource, Map<String, String> manifestAttributes) {
		files.put(filename, resource);
		
		if (manifestAttributes != null) {
			manifestDefinition.addEntryAttributes(filename, manifestAttributes);
		}
		return this;
	}

	public void write(File jarFile) throws IOException {
		FileOutputStream jarFileOut = new FileOutputStream(jarFile);
		try {
			JarOutputStream jarOut = 
				(manifestDefinition.isEmpty() ? new JarOutputStream(jarFileOut)
				                              : new JarOutputStream(jarFileOut, manifestDefinition.toManifest()));
			try {
				for (Map.Entry<String, Resource> eachFile : files.entrySet()) {
					String filename = eachFile.getKey();
					Resource resource = eachFile.getValue();
					jarOut.putNextEntry(new JarEntry(filename));
					InputStream resourceIn = resource.getInputStream();
					try {
						IOUtils.copy(resourceIn, jarOut);
					} finally {
						IOUtils.closeQuietly(resourceIn);
					}
					jarOut.closeEntry();
				}
			} finally {
				IOUtils.closeQuietly(jarOut);
			}
		} finally {
			IOUtils.closeQuietly(jarFileOut);
		}
	}
	
	public File writeToTemporaryFile() throws IOException {
		return writeToTemporaryFile(name, JAR_EXTENSION);
	}
	
	protected File writeToTemporaryFile(String prefix, String suffix) throws IOException {
		File tempJarFile = File.createTempFile(prefix, suffix);
		tempJarFile.deleteOnExit();
		write(tempJarFile);
		return tempJarFile;
	}
	
	public String getName() {
		return name;
	}
	
	private static class ManifestDefinition {
		private static final String GENERATED_MANIFEST_VERSION = "1.0";
		
		private final ReentrantLock lock = new ReentrantLock();
		
		private final ConcurrentMap<String, String> mainAttributes = new ConcurrentHashMap<String, String>();
		private final ConcurrentMap<String, Map<String, String>> entryAttributes = 
			new ConcurrentHashMap<String, Map<String, String>>();
		
		private ManifestDefinition addMainAttribute(String name, String value) {
			mainAttributes.put(name, value);
			return this;
		}
		
		private ManifestDefinition addEntryAttributes(String entryName, Map<String, String> attributes) {
			entryAttributes.putIfAbsent(entryName, 
					new ConcurrentHashMap<String, String>(checkNotNull(attributes).size()));
			entryAttributes.get(entryName).putAll(attributes);
			return this;
		}
		
		private boolean isEmpty() {
		    lock.lock();
			try {
				return (mainAttributes.isEmpty() && entryAttributes.isEmpty());
			} finally {
				lock.unlock();
			}
		}
		
		private Manifest toManifest() {
			Manifest manifest = new Manifest();
			Attributes manifestMainAttributes = manifest.getMainAttributes();
			manifestMainAttributes.putValue(Name.MANIFEST_VERSION.toString(), GENERATED_MANIFEST_VERSION);
			Map<String, Attributes> manifestEntries = manifest.getEntries();
		    lock.lock();
			try {
				for (Entry<String, String> mainAttribute : mainAttributes.entrySet()) {
					manifestMainAttributes.putValue(mainAttribute.getKey(), mainAttribute.getValue());
				}
				for (Entry<String, Map<String, String>> attributesForEntry : entryAttributes.entrySet()) {
					manifestEntries.put(attributesForEntry.getKey(), toAttributes(attributesForEntry.getValue()));
				}
			} finally {
				lock.unlock();
			}
			return manifest;
		}
		
		private static Attributes toAttributes(Map<String, String> values) {
			Attributes attributes = new Attributes(values.size());
			for (Entry<String, String> value : values.entrySet()) {
				attributes.putValue(value.getKey(), value.getValue());
			}
			return attributes;
		}
		
	}
}
