/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.test.support.onthefly;

import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Map.Entry;

import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.ci.DeploymentPackage;
import com.xebialabs.deployit.ci.artifact.ConfigurationFiles;
import com.xebialabs.deployit.ci.artifact.DeployableArtifact;
import com.xebialabs.deployit.ci.artifact.Ear;
import com.xebialabs.deployit.ci.artifact.Folder;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.SqlScript;
import com.xebialabs.deployit.ci.artifact.StaticContent;
import com.xebialabs.deployit.ci.artifact.War;

/**
 * Constructs a DAR, i.e. a JAR file that is a valid {@link DeploymentPackage}. Example:
 * 
 * <pre>
 * DarOnTheFly darOnTheFly = new DarOnTheFly("PetClinic", "1.0");
 * darOnTheFly.addFile("files/foo.txt", someFile) // will not be included in the package's manifest
 *            .addFile("bar.txt", someResource)
 *            ...
 *            // add CIs
 *            .addDeployableArtifact(SoLibrary.class, "solibs/lib.so", someFile)
 *            ...
 *            // also specify the CIs' 'name' property
 *            .addNamedDeployableArtifact("PetClinic-EJBs", EjbJar.jar, "enterprisearchive/myEjb.jar", someFile)
 *            ...
 *            // convenience methods
 *            .addEarOnTheFly(new EarOnTheFly("myEar"))
 *            .addWarOnTheFly(new WarOnTheFly("myWar"))
 *            .addConfigurationFile("config.xml", someFile)
 *            .addStaticContent("static.html", someResource)
 *            .addSqlScript("update-db.sql", someResource);
 * File dar = darOnTheFly.writeToTemporaryFile(); // something like PetClinic-1.0XX.zip
 * </pre>
 *
 */
public class DarOnTheFly extends JarOnTheFly {
	protected static final String DAR_EXTENSION = ".zip";
	
	// XXX: copied from ManifestBasedDeploymentPackageImportService...urgh...
	private static final String PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME = "Deployit-Package-Format-Version";
	private static final String PACKAGE_FORMAT_VERSION_NUMBER = "1.1";
	private static final String APPLICATION_ATTRIBUTE_NAME = "Ci-Application";
	private static final String VERSION_ATTRIBUTE_NAME = "Ci-Version";
	private static final String TYPE_ATTRIBUTE_NAME = "Ci-Type";
	private static final String NAME_ATTRIBUTE_NAME = "Ci-Name";
	
	private static final String JEE_ARTIFACT_DIRECTORY_NAME = "enterprisearchive";
	private static final String CONFIGURATION_FILES_DIRECTORY_NAME = "appconfig";
	private static final String STATIC_CONTENT_DIRECTORY_NAME = "staticcontent";
	private static final String SQL_SCRIPT_DIRECTORY_NAME = "sql";
	
	private final Map<String, FolderConfigurationItem> folders = Maps.newHashMapWithExpectedSize(2);

	public DarOnTheFly(String applicationName, String version) {
		super(applicationName + '-' + version);
		addManifestMainAttribute(PACKAGE_FORMAT_VERSION_ATTRIBUTE_NAME, PACKAGE_FORMAT_VERSION_NUMBER);
		addManifestMainAttribute(APPLICATION_ATTRIBUTE_NAME, applicationName);
		addManifestMainAttribute(VERSION_ATTRIBUTE_NAME, version);
		
		folders.put(CONFIGURATION_FILES_DIRECTORY_NAME, 
				    new FolderConfigurationItem(ConfigurationFiles.class, CONFIGURATION_FILES_DIRECTORY_NAME));
		folders.put(STATIC_CONTENT_DIRECTORY_NAME, 
			        new FolderConfigurationItem(StaticContent.class, STATIC_CONTENT_DIRECTORY_NAME));
	}
	
	public DarOnTheFly addDeployableArtifact(Class<? extends DeployableArtifact> configurationItemType,
			String filename, Resource resource) {
		addFile(filename, resource, makeAttributes(configurationItemType, null));
		return this;
	}
	
	private static Map<String, String> makeAttributes(Class<? extends DeployableArtifact> configurationItemType,
			String name) {
		boolean hasName = (name != null);
		Map<String, String> attributes = Maps.newHashMapWithExpectedSize(hasName ? 2 : 1);
		attributes.put(TYPE_ATTRIBUTE_NAME, configurationItemType.getSimpleName());
		if (hasName) {
			attributes.put(NAME_ATTRIBUTE_NAME, name);
		}
		return attributes;
	}
	
	public DarOnTheFly addNamedDeployableArtifact(String name, Class<? extends NamedDeployableArtifact> configurationItemType,
			String filename, Resource resource) {
		addFile(filename, resource, makeAttributes(configurationItemType, name));
		return this;
	}
	
	/* Convenience methods */
	
	public DarOnTheFly addWarOnTheFly(WarOnTheFly war) throws IOException {
		return addJeeArtifactOnTheFly(War.class, war, WarOnTheFly.WAR_EXTENSION);
	}
	
	public DarOnTheFly addEarOnTheFly(EarOnTheFly ear) throws IOException {
		return addJeeArtifactOnTheFly(Ear.class, ear, EarOnTheFly.EAR_EXTENSION);
	}
	
	private DarOnTheFly addJeeArtifactOnTheFly(Class<? extends NamedDeployableArtifact> configurationItemType,
			JarOnTheFly artifact, String artifactExtension) throws IOException {
		String artifactFilename = artifact.getName() + artifactExtension;
		addNamedDeployableArtifact(artifact.getName(), configurationItemType, 
				JEE_ARTIFACT_DIRECTORY_NAME + '/' + artifactFilename, 
				new FileSystemResource(artifact.writeToTemporaryFile()));
		return this;
	}
	
	public DarOnTheFly addConfigurationFile(String filename, Resource resource) {
		folders.get(CONFIGURATION_FILES_DIRECTORY_NAME).addEntry(filename, resource);
		return this;
	}
	
	public DarOnTheFly addStaticContent(String filename, Resource resource) {
		folders.get(STATIC_CONTENT_DIRECTORY_NAME).addEntry(filename, resource);
		return this;
	}
	
	public File writeToTemporaryFile() throws IOException {
		// "flush" the config file and static content buffers
		addFolders();
		return writeToTemporaryFile(name, DAR_EXTENSION);
	}	
	
	private void addFolders() {
		for (FolderConfigurationItem folder : folders.values()) {
			if (!folder.isEmpty()) {
				folder.addToDar();
			}
		}
	}
	
	public DarOnTheFly addSqlScript(String sqlScriptname, Resource sqlScript) {
		return addDeployableArtifact(SqlScript.class, SQL_SCRIPT_DIRECTORY_NAME + '/' + sqlScriptname, sqlScript);
	}
	
	private class FolderConfigurationItem {
		private final Class<? extends Folder> type;
		private final String dirname;
		private final Map<String, Resource> entries = Maps.newHashMap();
		
		private FolderConfigurationItem(Class<? extends Folder> type, String dirname) {
			this.type = type;
			this.dirname = dirname;
		}
		
		private void addEntry(String filename, Resource resource) {
			entries.put(filename, resource);
		}
		
		private boolean isEmpty() {
			return entries.isEmpty();
		}
		
		private void addToDar() {
			addManifestDirectoryAttributes(dirname, makeAttributes(type, null));
			for (Entry<String, Resource> entry : entries.entrySet()) {
				addFile(dirname + '/' + entry.getKey(), entry.getValue());
			}		
		}		

	}
	
}
