/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of Xebialabs Software and Documentation is subject to the Personal
 * License Agreement.
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * ‚ÄúDocumentation‚Äù means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the Xebialabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.test.support.onthefly;

import static com.xebialabs.deployit.test.support.utils.VelocityUtils.evaluateTemplate;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.Servlet;

import org.apache.velocity.VelocityContext;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;

/**
 * Constructs a WAR for testing, accepting multiple {@link Resource resources} and test {@link Servlet servlets}
 * as entries. Test servlets are automatically mapped to a path that is the servlet class' simple name,
 * so e.g. a {@code com.acme.package.TestJeeResourceServlet} will be bound to {@code http://myhost:myport/context-root/TestJeeResourceServlet}.
 * The actual WAR file is finally written to a temporary directory. Example:
 * 
 * <pre>
 * WarOnTheFly warOnTheFly = new WarOnTheFly("myWar");
 * warOnTheFly.addFile("foo", someFile);
 * warOnTheFly.addFile("bar", someResource);
 * ...
 * warOnTheFly.addServlet(FooServlet.class); // will be mapped to http://myhost:myport/context-root/FooServlet
 * warOnTheFly.addServlet(BarServlet.class);
 * ...
 * File war = warOnTheFly.writeToTemporaryFile(); // something like myWarXX.war
 * </pre>
 * 
 * See also &quot;<a href="http://blog.xebia.com/2009/12/14/middleware-integration-testing-with-junit-maven-and-vmware-part-2/">
 * Middleware integration testing with JUnit, Maven and VMware: part 2</a>&quot;.
 * 
 * @see JarOnTheFly
 */
public class WarOnTheFly extends JarOnTheFly {

	private String name;

	private Map<String, String> servlets;

	public WarOnTheFly(String name) {
		this.name = name;
		this.servlets = new HashMap<String, String>();
	}

	public void addServlet(Class<? extends Servlet> servletClass) {
		String servletName = servletClass.getSimpleName();
		String servletClassFilename = servletClass.getName().replace('.', '/') + ".class";
		addFile("WEB-INF/classes/" + servletClassFilename, new ClassPathResource(servletClassFilename));
		servlets.put(servletName, servletClass.getName());
	}

	public File writeToTemporaryFile() throws IOException {
		addWebXml();
		return writeToTemporaryFile(name, ".war");
	}

	public void addWebXml() {
		VelocityContext context = new VelocityContext();
		context.put("name", name);
		context.put("servlets", servlets);
		String webxml = evaluateTemplate(context, "com/xebialabs/deployit/test/support/onthefly/web.xml.vm");
		addFile("WEB-INF/web.xml", new ByteArrayResource(webxml.getBytes()));
	}

	public String getName() {
		return name;
	}

}
