package dsl

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.rest.test.api.Generator
import com.xebialabs.deployit.task.archive.ArchivedTask
import dsl.TaskState.TaskState
import org.joda.time.DateTime
import java.util.UUID

import ai.digital.deploy.tasker.common.{TaskMetadata, TaskType}

object TaskState extends Enumeration {
  type TaskState = Value
  val SUCCESSFUL, FAILED, ABORTED = Value
}

trait TaskUtils {
  this: Generator => // This trait can be used _only_ with Generator

  def createProvisioningTask(operation: String, template: String, envName: String, month: Int, durationInMinutes: Int, state: TaskState): Unit = {
    val startDate = new DateTime().minusMonths(month)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      t.parseState(state)
        .forCloudOperation(operation)
        .forCloudEnv(envName)
        .withCloudEnvTemplate(template)
        .startedAt(startDate)
        .completedAt(completionDate).withId(UUID.randomUUID().toString)
        .forDeploymentType(TaskType.CONTROL)
        .withDefaultBlock
    }
  }

  def createCurrentProvisoningTask(operation: String, template: String, envName: String, days: Int, durationInMinutes: Int, state: TaskState): Unit = {
    val startDate = new DateTime().minusDays(days)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      t.parseState(state)
        .forCloudOperation(operation)
        .forCloudEnv(envName)
        .withCloudEnvTemplate(template)
        .startedAt(startDate).completedAt(completionDate)
        .withId(UUID.randomUUID().toString)
        .forDeploymentType(TaskType.CONTROL)
        .withDefaultBlock
    }
  }

  def createTask(app: BaseConfigurationItem, version: String, env: BaseConfigurationItem, month: Int, durationInMinutes: Int, state: TaskState, taskType: TaskType): Unit = {
    val startDate = new DateTime().minusMonths(month)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      t.parseState(state)
        .forApp(app.getName, app.get$internalId)
        .forVersion(version)
        .forEnv(env.getName, env.get$internalId)
        .startedAt(startDate)
        .completedAt(completionDate)
        .withId(UUID.randomUUID().toString)
        .forDeploymentType(taskType)
        .withDefaultBlock
    }
  }

  def createControlTask(taskName: String, target: BaseConfigurationItem,
                        days: Int, durationInMinutes: Int, state: TaskState, description: String = null): Unit = {
    val startDate = new DateTime().minusDays(days)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      t.parseState(state)
        .startedAt(startDate)
        .completedAt(completionDate)
        .withId(UUID.randomUUID().toString)
        .forDeploymentType(TaskType.CONTROL)
        .withDescription(description)
        .addMetadata(
          (TaskMetadata.CONTROL_TASK_TARGET_CI, target.getId),
          (TaskMetadata.CONTROL_TASK_TARGET_INTERNAL_CI, target.get$internalId().toString),
          (TaskMetadata.CONTROL_TASK_TARGET_SECURED_CI, target.get$securedCi().toString),
          (TaskMetadata.TASK_NAME, taskName)
        )
        .withDefaultBlock
    }
  }

  def createCurrentTask(app: BaseConfigurationItem, version: String, env: BaseConfigurationItem, days: Int, durationInMinutes: Int, state: TaskState, taskType: TaskType, taskFiller: ArchivedTask => ArchivedTask = _.withDefaultBlock): Unit = {
    val startDate = new DateTime().minusDays(days)
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      taskFiller.apply(t.parseState(state)
        .forApp(app.getName, app.get$internalId)
        .forVersion(version)
        .forEnv(env.getName, env.get$internalId)
        .startedAt(startDate)
        .completedAt(completionDate)
        .withId(UUID.randomUUID.toString)
        .forDeploymentType(taskType))
    }
  }

  def createTaskWithDependencies(app: BaseConfigurationItem, version: String, env: BaseConfigurationItem, startDate: DateTime, durationInMinutes: Int,
                                 state: TaskState, taskType: TaskType, dependencies: Map[String, String]): Unit = {
    val completionDate = startDate.plusMinutes(durationInMinutes)

    task { t =>
      val preDepsTask = t.parseState(state)
        .forApp(app.getName, app.get$internalId)
        .forVersion(version)
        .forEnv(env.getName, env.get$internalId)
        .startedAt(startDate)
        .completedAt(completionDate)
        .withId(UUID.randomUUID().toString)
        .forDeploymentType(taskType)
        .withDefaultBlock
      dependencies.foldLeft(preDepsTask) { case (acc, (a, v)) => acc.withDependency(a, v) }
    }
  }

  implicit class ExtendedArchivedTask(task: ArchivedTask) {
    def parseState(state: TaskState): ArchivedTask = {
      state match {
        case TaskState.SUCCESSFUL => task.success
        case TaskState.FAILED => task.successWithFails
        case TaskState.ABORTED => task.failed
      }
    }
  }

}
