package dsl

import com.google.common.base.Charsets
import com.google.common.io.{Files => CommonsFiles}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.test.v3.ConfigurationFiles
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.local.LocalFile
import dsl.{Creator => DslCreator}

import java.io.File
import java.nio.file.Files

trait EarUtils {

  private val tempDir: File = Files.createTempDirectory("ear-utils").toFile

  def createEmptyPackage(c: DslCreator, appName: String, nrPackages: Int): Unit = {
    createPackage(c, appName, nrPackages) { _ =>
      Nil
    }
  }

  def createPackageWithDummyEar(c: DslCreator, appName: String, nrPackages: Int, packageSize: Int, earName: String): Unit = {
    createPackage(c, appName, nrPackages) { dp =>
      for (j <- 0 until packageSize) {
        createDummyEarFile(c, earName + j, dp)
      }
      Nil
    }
  }

  def createPackageWithDummyEarWithFailingStep(c: DslCreator, appName: String, nrPackages: Int, packageSize: Int, earName: String): Unit = {
    createPackage(c, appName, nrPackages) { dp =>
      createDummyEarFileWithFailingStep(c, earName + "failing" + 0, dp)
      for (j <- 1 until packageSize) {
        createDummyEarFile(c, earName + j, dp)
      }
      Nil
    }
  }

  def createPackageWithDummyEarWithProperties(c: DslCreator, appName: String, nrPackages: Int, packageSize: Int, earName: String): Unit = {
    createPackage(c, appName, nrPackages) { dp =>
      for (j <- 0 until packageSize) {
        createDummyEarWithPropertiesFile(c, earName + j, dp)
      }
      Nil
    }
  }

  def createPackageWithDummyEarAndConfigurationFile(c: DslCreator, appName: String, nrPackages: Int, packageSize: Int, earName: String): Unit = {
    createPackage(c, appName, nrPackages) { dp =>
      val configName = appName + "config"
      for (j <- 0 until packageSize) {
        createDummyEarFile(c, earName + j, dp)
        createConfigFile(c, configName + j, dp)
      }
      Nil
    }
  }


  def createConfigFile(c: DslCreator, configName: String, dp: DeploymentPackage): ConfigurationFiles = {
    c.configurationFile(configName, dp) { c =>
      c.setFile(LocalFile.valueOf({
        val directory = new File(tempDir, configName)
        if (directory.mkdir()) {
          createFile(directory, configName + ".config")
        }
        directory
      }))
    }
  }

  def createFile(tempDir: File, fileName: String): OverthereFile = {
    LocalFile.valueOf({
      val file: File = new File(tempDir, fileName)
      file.createNewFile()
      CommonsFiles.asCharSink(file, Charsets.UTF_8).write("**********")
      file
    })
  }

  def createPackage(c: DslCreator, appName: String, nrPackages: Int)(deployables: DeploymentPackage => List[_ <: Deployable]): Unit = {
    val app: Application = c.application(appName)
    for (i <- 0 until nrPackages) {
      val version = i + ".0"
      app.setLastVersion(version)
      c.deploymentPackage(version, app) { dp =>
        deployables.apply(dp)
      }
    }
  }

  def createDummyEarFile(c: DslCreator, earName: String, dp: DeploymentPackage): Deployable = {
    c.ear(earName, dp) { e =>
      e.setFile(createFile(tempDir, earName + ".ear"))
    }
  }

  def createDummyEarFileWithFailingStep(c: DslCreator, earName: String, dp: DeploymentPackage): Deployable = {
    c.ear(earName, dp) { e =>
      e.setFile(createFile(tempDir, earName + ".ear"))
      e.setFailToDeployOnDummyJeeServer(true)
    }
  }

  def createDummyEarWithPropertiesFile(c: DslCreator, earName: String, dp: DeploymentPackage): Deployable = {
    c.earWithProperties(earName, dp) { e =>
      e.setFile(createFile(tempDir, earName + ".ear"))
    }
  }

  def createPackageWithDependencies(c: DslCreator, appName: String, version: String, dependencies: Map[String, String]): Unit = {
    val app = c.application(appName)
    app.setLastVersion(version)
    c.deploymentPackage(version, app, dependencies) { _ => Nil }
  }

}
