/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit;

import static com.xebialabs.deployit.DeployitConfig.SPRING_CONFIG;
import static com.xebialabs.deployit.DeployitConfig.SPRING_CTX_CFG_LOCATION_KEY;
import static org.springframework.web.context.support.WebApplicationContextUtils.getWebApplicationContext;

import java.net.InetAddress;
import java.net.ServerSocket;
import java.util.Map;
import java.util.concurrent.CountDownLatch;

import javax.jcr.Repository;

import org.apache.jackrabbit.j2ee.JcrRemotingServlet;
import org.apache.james.mime4j.storage.DefaultStorageProvider;
import org.jboss.resteasy.plugins.server.servlet.HttpServletDispatcher;
import org.jboss.resteasy.plugins.server.servlet.ResteasyBootstrap;
import org.jboss.resteasy.plugins.spring.SpringContextLoaderListener;
import org.mortbay.jetty.bio.SocketConnector;
import org.mortbay.jetty.security.SslSocketConnector;
import org.mortbay.jetty.servlet.Context;
import org.mortbay.jetty.servlet.FilterHolder;
import org.mortbay.jetty.servlet.ServletHolder;
import org.mortbay.thread.BoundedThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.filter.DelegatingFilterProxy;

import com.google.common.base.Strings;
import com.xebialabs.deployit.core.rest.resteasy.Mime4jStorageProvider;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.event.EventCallback;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.jetty.ClassPathResourceContentServlet;
import com.xebialabs.deployit.jetty.MethodOverrideFilter;
import com.xebialabs.deployit.jetty.RequestHeadersEncodedAsParametersFilter;
import com.xebialabs.deployit.plugin.api.boot.PluginBooter;

public class Server {
	private final DeployitConfig deployItConfiguration;

	private org.mortbay.jetty.Server jettyServer;
	private Context contextRoot;
	private static volatile CountDownLatch shutdownLatch = new CountDownLatch(1);

	public static void start(DeployitOptions deployitOptions) {
		DeployitConfiguration.load();
		Server server = new Server(DeployitConfiguration.getInstance(), deployitOptions);
		server.start();
	}

	public Server(DeployitConfig configuration, final DeployitOptions deployitOptions) {
		deployItConfiguration = configuration;
		if (portIsAvailable()) {
			setupJetty(deployitOptions);
            bootPlugins();
			setupSpring();
			setupRest();
		} else {
			logger.error("Cannot start Deployit server, port " + deployItConfiguration.getHttpPort() + " already in use.");
			logger.error("Perhaps another instance of Deployit is running. If so, please shutdown the other instance of Deployit.");
			System.exit(1);
		}
	}

    private boolean portIsAvailable() {
		try {
			ServerSocket socket = new ServerSocket(deployItConfiguration.getHttpPort());
			socket.close();
		} catch (Exception e) {
			return false;
		}
		return true;
	}

	private void setupJetty(final DeployitOptions deployitOptions) {
		int port = deployItConfiguration.getHttpPort();
		String webContextRoot = deployItConfiguration.getWebContextRoot();
		String documentRootPackage = DeployitConfig.DEFAULT_WEBCONTENT_PACKAGE;

		if (logger.isDebugEnabled())
			logger.debug("Setting up Jetty server on port {} with context root {} using package {} for document root.",
					new Object[] {port,webContextRoot, documentRootPackage});

		jettyServer = new org.mortbay.jetty.Server();
		SocketConnector connector = new SocketConnector();
		if (deployItConfiguration.isSsl()) {
			logger.debug("Setting up Jetty to use SSL");
			SslSocketConnector sslConnector = new SslSocketConnector();
			// TODO: Define a property in deployit.conf
			connector.setMaxIdleTime(30000);
			sslConnector.setKeystore(deployItConfiguration.getKeyStorePath());
			sslConnector.setPassword(deployItConfiguration.getKeyStorePassword());
			sslConnector.setKeyPassword(deployItConfiguration.getKeyStoreKeyPassword());
			String protocol = deployItConfiguration.getSslProtocol();
			if (!Strings.nullToEmpty(protocol).trim().isEmpty()) {
				sslConnector.setProtocol(protocol);
			}

			connector = sslConnector;
		}
		connector.setHost(deployItConfiguration.getHttpBindAddress());
		connector.setPort(port);
		logger.info("Connector listen {} on {}:{}", new Object[]{deployItConfiguration.isSsl() ? "secure" : "no-secure", connector.getHost(), connector.getPort() });
		jettyServer.addConnector(connector);

		BoundedThreadPool btp = new BoundedThreadPool();
		btp.setMinThreads(deployItConfiguration.getMinThreads());
		btp.setMaxThreads(deployItConfiguration.getMaxThreads());
		jettyServer.setThreadPool(btp);

		jettyServer.setGracefulShutdown(1000);

		contextRoot = new Context(jettyServer, webContextRoot, Context.SESSIONS);
        contextRoot.addFilter(RequestHeadersEncodedAsParametersFilter.class, "/deployit/reports/*", Context.SESSIONS);
        contextRoot.addFilter(MethodOverrideFilter.class, "*", Context.SESSIONS);


		ServletHolder documentHolder = new ServletHolder();
		documentHolder.setServlet(new ClassPathResourceContentServlet(documentRootPackage));
		contextRoot.addServlet(documentHolder, "/");

		if (deployitOptions.isTestModeEnabled()) {
			setupJackRabbitAccess();
		}
	}

	private void setupJackRabbitAccess() {
		@SuppressWarnings("serial")
        ServletHolder remotingServletHolder = new ServletHolder(new JcrRemotingServlet() {
			@Override
			protected Repository getRepository() {
				WebApplicationContext context = WebApplicationContextUtils.getWebApplicationContext(getServletContext());
				return (Repository) context.getBean("jcrRepository");

			}
		});
		remotingServletHolder.setName("RepositoryAccessServlet");
		String repositoryPrefix = "/repository-test";
		remotingServletHolder.setInitParameter("resource-path-prefix", repositoryPrefix);
		contextRoot.addServlet(remotingServletHolder, repositoryPrefix + "/*");

		// Allows you to the access the JCR repository via a web browser:
		// URL: http://localhost:4516/repository/default/
		@SuppressWarnings("serial")
		ServletHolder webdavServletHolder = new ServletHolder(new JcrRemotingServlet() {
			@Override
			public Repository getRepository() {
				WebApplicationContext context = getWebApplicationContext(getServletContext());
				return (Repository) context.getBean("jcrRepository");
			}
		});
		webdavServletHolder.setName("Webdav"); // the name given to this servlet in jackrabbit-webapp-2.1.0.war/WEB-INF/web.xml
		repositoryPrefix = "/repository";
		webdavServletHolder.setInitParameter("resource-path-prefix", repositoryPrefix);
        webdavServletHolder.setInitParameter("missing-auth-mapping", "guestcredentials");
		contextRoot.addServlet(webdavServletHolder, repositoryPrefix + "/*");
	}

    private void bootPlugins() {
        PluginBooter.boot();
    }

	@SuppressWarnings("unchecked")
	private void setupSpring() {
		Map<String, String> params = contextRoot.getInitParams();
		params.put(SPRING_CTX_CFG_LOCATION_KEY, SPRING_CONFIG);
		if (logger.isDebugEnabled()) {
			logger.debug("Using Spring configuration - " + params.get(SPRING_CTX_CFG_LOCATION_KEY));
		}

//		contextRoot.setInitParams(params);
//		contextRoot.addEventListener(new ContextLoaderListener() {
//			@Override
//			public void contextInitialized(ServletContextEvent event) {
//				try {
//					super.contextInitialized(event);
//				} catch (Exception exc) {
//					logger.error("Exception received while initializing the Spring context in the server. Exiting...", exc);
//					System.exit(1);
//				}
//			}
//		});

		FilterHolder filter = new FilterHolder(DelegatingFilterProxy.class);
		filter.setName("springSecurityFilterChain");
		contextRoot.addFilter(filter, "/deployit/*", 1);
	}

	@SuppressWarnings("unchecked")
	private void setupRest() {
		contextRoot.addEventListener(new ResteasyBootstrap());
		contextRoot.addEventListener(new SpringContextLoaderListener());

		ServletHolder servletHolder = new ServletHolder(HttpServletDispatcher.class);
		contextRoot.getInitParams().put("resteasy.servlet.mapping.prefix", "/deployit");
		contextRoot.getInitParams().put("resteasy.document.expand.entity.references", "false");
//		servletHolder.setInitParameter("contextConfigLocation", "classpath*:springmvc-servlet.xml");
		contextRoot.addServlet(servletHolder, "/deployit/*");
		contextRoot.setResourceBase(".");

//		buildDefaultResteasyProviderFactory()
//                .addExceptionMapper(DeployitExceptionMapper.class, UnhandledExceptionMapper.class)
//                .addPreProcessInterceptor(OpenJcrSessionDuringRequestInterceptor.class)
//                .addPostProcessInterceptor(OpenJcrSessionDuringRequestInterceptor.class)
//                .build();
		DefaultStorageProvider.setInstance(new Mime4jStorageProvider());
	}

	public void start() {
		try {
			if (logger.isDebugEnabled())
				logger.debug("Starting Jetty");

			jettyServer.start();

			if (logger.isDebugEnabled())
				logger.debug("Jetty started");
			logger.info("The Deployit server has started.");

			createShutdownEventThreadAndShutdownHook();

			String host = new StringBuilder()
					.append(deployItConfiguration.isSsl() ? "https" : "http")
					.append("://").append(InetAddress.getLocalHost().getHostName())
					.append(":").append(deployItConfiguration.getHttpPort())
					.append(deployItConfiguration.getWebContextRoot()).toString();
			logger.info("You can now point your browser to {} or start the command line interface with \"cli.sh\" or \"cli.cmd\"", host);
		} catch (Exception e) {
			logger.error("Error occured while starting Jetty", e);
		}
	}

	private synchronized void stop() {
		if (jettyServer.isRunning()) {
			logger.info("Shutting down the Deployit server.");
			try {
				jettyServer.stop();
				logger.info("The Deployit server has shut down.");
			} catch (Exception e) {
				logger.error("Error occurred while trying to stop Jetty", e);
			}
		}
	}

	private void createShutdownEventThreadAndShutdownHook() {
		Thread shutdownEventThread = new Thread(new Runnable() {
			public void run() {
				try {
                	shutdownLatch.await();
                } catch (InterruptedException e) {
                	Thread.currentThread().interrupt();
                }
				logger.trace("Shutting down the Deployit server because a shutdown event was received.");
				stop();
			}
		});
		shutdownEventThread.start();
		
		Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
			@Override
			public void run() {
				logger.trace("Shutting down the Deployit server because the process is being terminated.");
				stop();
			}
		}));
	}

	static {
		EventBus.registerForEvent(ShutdownEvent.class, new EventCallback<ShutdownEvent>() {
			@Override
			public void receive(final ShutdownEvent event) {
				logger.trace("Received shutdown event.");
                shutdownLatch.countDown();
			}
		});
	}

	private static final Logger logger = LoggerFactory.getLogger(Server.class);

}
