/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit;

import com.google.common.base.Strings;
import com.xebialabs.deployit.core.rest.resteasy.OpenJcrSessionDuringRequestInterceptor;
import com.xebialabs.deployit.event.EventBus;
import com.xebialabs.deployit.event.EventCallback;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.jetty.ClassPathResourceContentServlet;
import com.xebialabs.deployit.jetty.MethodOverrideFilter;
import com.xebialabs.deployit.jetty.RequestHeaderOverrideFilter;
import com.xebialabs.deployit.plugin.api.boot.PluginBooter;
import org.apache.jackrabbit.j2ee.JcrRemotingServlet;
import org.mortbay.jetty.bio.SocketConnector;
import org.mortbay.jetty.security.SslSocketConnector;
import org.mortbay.jetty.servlet.Context;
import org.mortbay.jetty.servlet.FilterHolder;
import org.mortbay.jetty.servlet.ServletHolder;
import org.mortbay.thread.BoundedThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.ContextLoaderListener;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.filter.DelegatingFilterProxy;
import org.springframework.web.servlet.DispatcherServlet;

import javax.jcr.Repository;
import javax.servlet.ServletContextEvent;
import java.net.ServerSocket;
import java.util.Map;
import java.util.concurrent.CountDownLatch;

import static com.xebialabs.deployit.DeployItConfiguration.SPRING_CONFIG;
import static com.xebialabs.deployit.DeployItConfiguration.SPRING_CTX_CFG_LOCATION_KEY;
import static com.xebialabs.deployit.core.api.resteasy.ResteasyHelper.buildDefaultResteasyProviderFactory;
import static org.springframework.web.context.support.WebApplicationContextUtils.getWebApplicationContext;

public class Server {
	private final DeployItConfiguration deployItConfiguration;

	private org.mortbay.jetty.Server jettyServer;
	private Context contextRoot;
	private static volatile CountDownLatch shutdownLatch = new CountDownLatch(1);

	static {
		EventBus.registerForEvent(ShutdownEvent.class, new EventCallback<ShutdownEvent>() {
			@Override
			public void receive(final ShutdownEvent event) {
				requestShutdown();
			}
		});
	}

	public static void start(DeployitOptions deployitOptions) {
		DeployItConfiguration configuration = DeployItConfiguration.loadForUse();
		Server server = new Server(configuration, deployitOptions);
		server.start();
	}

	public Server(DeployItConfiguration configuration, final DeployitOptions deployitOptions) {
		deployItConfiguration = configuration;
		if (portIsAvailable()) {
			setupJetty(deployitOptions);
            bootPlugins();
			setupSpring();
			setupRest();
		} else {
			logger.error("Cannot start Deployit server, port " + deployItConfiguration.getHttpPort() + " already in use.");
			logger.error("Perhaps another instance of Deployit is running. If so, please shutdown the other instance of Deployit.");
			System.exit(1);
		}
	}

    private void bootPlugins() {
        PluginBooter.boot();
    }

    private boolean portIsAvailable() {
		try {
			ServerSocket socket = new ServerSocket(deployItConfiguration.getHttpPort());
			socket.close();
		} catch (Exception e) {
			return false;
		}
		return true;
	}

	public void start() {
		try {
			if (logger.isDebugEnabled())
				logger.debug("Starting Jetty");

			jettyServer.start();

			if (logger.isDebugEnabled())
				logger.debug("Jetty started");
			logger.info("The Deployit server has started.");

			logger.info("You can now point your browser to " + (deployItConfiguration.isSsl() ? "https" : "http") + "://localhost:"
			        + deployItConfiguration.getHttpPort() + " or start the command line interface with \"cli.sh\" or \"cli.cmd\"");

			Thread shutdownServerThread = new Thread(new Runnable() {
				public void run() {
					try {
						shutdownLatch.await();
					} catch (InterruptedException e) {
						Thread.currentThread().interrupt();
					}
					if (logger.isDebugEnabled()) {
						logger.debug("Stopping Jetty.");
					}
					try {
						jettyServer.stop();
						logger.info("Server stopped.");
					} catch (Exception e) {
						logger.error("Error occurred while trying to stop Jetty", e);
					}
				}
			});

			shutdownServerThread.start();

		} catch (Exception e) {
			logger.error("Error occured while starting Jetty", e);
		}
	}

	private void setupJetty(final DeployitOptions deployitOptions) {
		int port = deployItConfiguration.getHttpPort();
		String webContextRoot = DeployItConfiguration.DEFAULT_WEB_CONTEXT_ROOT;
		String documentRootPackage = DeployItConfiguration.DEFAULT_WEBCONTENT_PACKAGE;

		if (logger.isDebugEnabled())
			logger.debug("Setting up Jetty server on port " + port + " with context root " + webContextRoot + " using package " + documentRootPackage
			        + " for document root.");

		if (deployItConfiguration.isSsl()) {
			if (logger.isDebugEnabled()) {
				logger.debug("Setting up Jetty to use SSL");
			}
			jettyServer = new org.mortbay.jetty.Server();
			SslSocketConnector connector = new SslSocketConnector();
			// TODO: Define a deployit properties
			connector.setHost(System.getProperty("jetty.host", "localhost"));
			connector.setPort(port);
			logger.info("Connector listen secure on " + connector.getHost() + ":" + connector.getPort());
			connector.setMaxIdleTime(30000);
			connector.setKeystore(deployItConfiguration.getKeyStorePath());
			connector.setPassword(deployItConfiguration.getKeyStorePassword());
			connector.setKeyPassword(deployItConfiguration.getKeyStoreKeyPassword());
			String protocol = deployItConfiguration.getSslProtocol();
			if (!Strings.nullToEmpty(protocol).trim().isEmpty()) {
				connector.setProtocol(protocol);
			}
			jettyServer.addConnector(connector);
		} else {
			jettyServer = new org.mortbay.jetty.Server();
			SocketConnector connector = new SocketConnector();

			// TODO: Define a deployit properties
			connector.setHost(System.getProperty("jetty.host"));
			connector.setPort(port);
			logger.info("Connector listen no-secure on " + connector.getHost() + ":" + connector.getPort());
			jettyServer.addConnector(connector);
		}

		BoundedThreadPool btp = new BoundedThreadPool();
		btp.setMinThreads(deployItConfiguration.getMinThreads());
		btp.setMaxThreads(deployItConfiguration.getMaxThreads());
		jettyServer.setThreadPool(btp);

		contextRoot = new Context(jettyServer, webContextRoot, Context.SESSIONS);
		contextRoot.addFilter(MethodOverrideFilter.class, "*", Context.SESSIONS);
		contextRoot.addFilter(RequestHeaderOverrideFilter.class, "*", Context.SESSIONS);

		ServletHolder documentHolder = new ServletHolder();
		documentHolder.setServlet(new ClassPathResourceContentServlet(documentRootPackage));
		contextRoot.addServlet(documentHolder, "/");

		if (deployitOptions.isTestModeEnabled()) {
			setupJackRabbitAccess();
		}
	}

	private void setupJackRabbitAccess() {
		@SuppressWarnings("serial")
        ServletHolder remotingServletHolder = new ServletHolder(new JcrRemotingServlet() {
			@Override
			protected Repository getRepository() {
				WebApplicationContext context = WebApplicationContextUtils.getWebApplicationContext(getServletContext());
				return (Repository) context.getBean("jcrRepository");

			}
		});
		remotingServletHolder.setName("RepositoryAccessServlet");
		String repositoryPrefix = "/repository-test";
		remotingServletHolder.setInitParameter("resource-path-prefix", repositoryPrefix);
		contextRoot.addServlet(remotingServletHolder, repositoryPrefix + "/*");

		// Allows you to the access the JCR repository via a web browser:
		// URL: http://localhost:4516/repository/default/
		@SuppressWarnings("serial")
		ServletHolder webdavServletHolder = new ServletHolder(new JcrRemotingServlet() {
			@Override
			public Repository getRepository() {
				WebApplicationContext context = getWebApplicationContext(getServletContext());
				return (Repository) context.getBean("jcrRepository");
			}
		});
		webdavServletHolder.setName("Webdav"); // the name given to this servlet in jackrabbit-webapp-2.1.0.war/WEB-INF/web.xml
		repositoryPrefix = "/repository";
		webdavServletHolder.setInitParameter("resource-path-prefix", repositoryPrefix);
        webdavServletHolder.setInitParameter("missing-auth-mapping", "guestcredentials");
		contextRoot.addServlet(webdavServletHolder, repositoryPrefix + "/*");
	}

	@SuppressWarnings("unchecked")
	private void setupSpring() {
		Map<String, String> params = contextRoot.getInitParams();
		params.put(SPRING_CTX_CFG_LOCATION_KEY, SPRING_CONFIG);
		if (logger.isDebugEnabled()) {
			logger.debug("Using Spring configuration - " + params.get(SPRING_CTX_CFG_LOCATION_KEY));
		}

//		contextRoot.setInitParams(params);
		contextRoot.addEventListener(new ContextLoaderListener() {
			@Override
			public void contextInitialized(ServletContextEvent event) {
				try {
					super.contextInitialized(event);
				} catch (Exception exc) {
					logger.error("Exception received while initializing the Spring context in the server. Exiting...", exc);
					System.exit(1);
				}
			}
		});

		FilterHolder filter = new FilterHolder(DelegatingFilterProxy.class);
		filter.setName("springSecurityFilterChain");
		contextRoot.addFilter(filter, "/deployit/*", 1);
	}

	@SuppressWarnings("unchecked")
	private void setupRest() {
		ServletHolder servletHolder = new ServletHolder(DispatcherServlet.class);
		contextRoot.getInitParams().put("resteasy.servlet.mapping.prefix", "/deployit");
		servletHolder.setInitParameter("contextConfigLocation", "classpath*:springmvc-servlet.xml");
		contextRoot.addServlet(servletHolder, "/deployit/*");
		contextRoot.setResourceBase(".");
		
		buildDefaultResteasyProviderFactory()
                .addExceptionMapper(DeployitExceptionMapper.class, UnhandledExceptionMapper.class)
                .addPreProcessInterceptor(OpenJcrSessionDuringRequestInterceptor.class)
                .addPostProcessInterceptor(OpenJcrSessionDuringRequestInterceptor.class)
                .build();
	}

	public static void requestShutdown() {
		logger.info("Received message to shutdown the server.");
		shutdownLatch.countDown();
	}

	public static boolean hasShutdownBeenrequested() {
		return shutdownLatch.getCount() == 0;
	}

	private static final Logger logger = LoggerFactory.getLogger(Server.class);
}
