package com.xebialabs.deployit;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import javax.crypto.SecretKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.setup.Setup;
import com.xebialabs.deployit.util.DeployitKeyStoreException;
import com.xebialabs.deployit.util.DeployitKeys;
import com.xebialabs.deployit.util.PasswordEncrypter;

public class XLDeployServer {
    private static final Logger logger = LoggerFactory.getLogger(XLDeployServer.class);

    private static final List<File> DEPRECATED_SYSTEM_CONFIG_FILES = Arrays.asList(
            new File("conf", "system.conf"),
            new File("conf", "maven.conf"),
            new File("conf", "extensions.conf"),
            new File("conf", "planner.conf"),
            new File("conf", "scheduler.conf"));

    private final ServerLaunchOptions launchOptions;
    private final ServerConfigFile serverConfigFile;
    private boolean isConfigAndPasswordInitDone = false;

    public XLDeployServer(ServerLaunchOptions launchOptions, ServerConfigFile serverConfigFile) {
        this.launchOptions = launchOptions;
        this.serverConfigFile = serverConfigFile;
    }

    void prepareConfigs() throws IOException {
        logger.info("Preparing configuration files before booting the application");
        checkDeprecatedSystemConf();
        if (launchOptions.isDoSetup()) {
            startSetup(launchOptions, serverConfigFile);
        } else {
            if (!serverConfigFile.exists()) {
                logger.warn("Configuration not found...");
                startSetup(launchOptions, serverConfigFile);
            } else if (launchOptions.isReinitialize()) {
                logger.warn("Not re-initializing as the -setup flag is missing.");
            }
            if (!isConfigAndPasswordInitDone) {
                loadConfigAndInitPasswordEncryptionKey();
            }
        }
    }

    private void checkDeprecatedSystemConf() {
        DEPRECATED_SYSTEM_CONFIG_FILES.forEach(configFile -> {
            if (configFile.exists()) {
                throw new RuntimeException("You are not allowed to have a " + configFile.getName() + " file on the 'conf' folder. ");
            }
        });
    }

    private static void startSetup(ServerLaunchOptions launchOptions, ServerConfigFile serverConfigFile) {
        new Setup("XL Deploy Server", serverConfigFile, launchOptions).run();
    }

    void loadConfigAndInitPasswordEncryptionKey() {
        ServerConfiguration config = serverConfigFile.loadConfig(false, false, true);

        String repositoryKeyStorePassword;
        if (launchOptions.getRepositoryKeystorePassword() != null) {
            repositoryKeyStorePassword = launchOptions.getRepositoryKeystorePassword();
        } else {
            repositoryKeyStorePassword = config.getRepositoryKeyStorePassword();
        }
        try {
            SecretKey passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(repositoryKeyStorePassword);
            PasswordEncrypter.init(passwordEncryptionKey);
        } catch (DeployitKeyStoreException e) {
            throw new RuntimeException("Could not load the encryption key. The server will not start.");
        }
        config.loadEncryptedPasswords();
        serverConfigFile.saveDirtyConfiguration(config);
        ServerConfiguration.setInstance(config);
        ServerLaunchOptions.setInstance(launchOptions);
        isConfigAndPasswordInitDone = true;
    }
}
