package com.xebialabs.deployit.server.jetty;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.XldRequestLog;
import com.xebialabs.deployit.util.SslFactoryBuilder;
import org.eclipse.jetty.http.HttpCompliance;
import org.eclipse.jetty.http.UriCompliance;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.SecureRequestCustomizer;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.embedded.jetty.JettyServerCustomizer;

import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.xebialabs.deployit.ServerConfiguration.DEFAULT_STOP_TIMEOUT;

class JettyServerConnector {

    public static JettyServerCustomizer create(ServerConfiguration serverConfiguration) {
        return jettyServer -> {
            jettyServer.setStopTimeout(DEFAULT_STOP_TIMEOUT);

            ServerConnector connector;
            if (serverConfiguration.isSsl()) {
                logger.debug("Setting up Jetty to use SSL");
                SslContextFactory.Server sslContextFactory = SslFactoryBuilder.build(serverConfiguration);
                HttpConnectionFactory httpConnectionFactory = getHttpConnectionFactory();
                connector = new ServerConnector(jettyServer, sslContextFactory, httpConnectionFactory);
            } else {
                HttpConnectionFactory httpConnectionFactory = getHttpConnectionFactory();
                connector = new ServerConnector(jettyServer, httpConnectionFactory);
            }

            connector.setIdleTimeout(serverConfiguration.getHttpIdleTimeoutMillis());
            connector.setHost(serverConfiguration.getHttpBindAddress());
            connector.setPort(serverConfiguration.getHttpPort());

            jettyServer.setConnectors(new ServerConnector[]{connector});

            // setupAccessLog - this should be replaced either by logback-access once it is released or CustomRequestLog
            final XldRequestLog logWriter = new XldRequestLog("conf/logback-access.xml");
            jettyServer.setRequestLog(logWriter);
            logger.info("Connector listen {} on {}:{}", serverConfiguration.isSsl() ? "secure" : "no-secure", connector.getHost(), connector.getPort());
        };
    }

    private static HttpConnectionFactory getHttpConnectionFactory() {
        HttpConfiguration configuration = new HttpConfiguration();
        configuration.setUriCompliance(UriCompliance.from(Set.of(
                UriCompliance.Violation.AMBIGUOUS_PATH_SEPARATOR,
                UriCompliance.Violation.AMBIGUOUS_PATH_ENCODING)));
        configuration.addCustomizer(new SecureRequestCustomizer(
                true,
                TimeUnit.SECONDS.convert(365, TimeUnit.DAYS),
                true
        ));
        HttpConnectionFactory httpConnectionFactory = new HttpConnectionFactory(configuration);
        httpConnectionFactory.getHttpConfiguration().setHttpCompliance(HttpCompliance.RFC2616);
        httpConnectionFactory.getHttpConfiguration().setSendServerVersion(false);
        return httpConnectionFactory;
    }

    private static final Logger logger = LoggerFactory.getLogger(JettyServerConnector.class);
}
