package com.xebialabs.deployit.server.jetty;

import ai.digital.deploy.core.common.XldServerPaths;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.booter.local.GlobalContextManager;
import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.local.message.MessageService;
import com.xebialabs.plugin.classloader.PluginClassLoader;
import com.xebialabs.plugin.manager.startup.PluginSynchronizer;
import com.xebialabs.plugin.manager.startup.SynchronizationCondition;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.embedded.jetty.JettyServletWebServerFactory;
import org.springframework.boot.web.server.WebServerFactoryCustomizer;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import java.net.ServerSocket;
import java.security.Security;

@Component
public class JettyServer implements WebServerFactoryCustomizer<JettyServletWebServerFactory> {

    @Autowired
    private MessageService deployMessageService;

    @Autowired
    private ServerConfiguration serverConfiguration;

    @Autowired
    private PluginSynchronizer pluginSynchronizer;

    @Autowired
    private GlobalContextManager globalContextManager;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    private final String DEPLOYIT = "deployit";

    @Override
    public void customize(JettyServletWebServerFactory factory) {
        int port = serverConfiguration.getHttpPort();
        String webContextRoot = serverConfiguration.getWebContextRoot();

        Security.addProvider(new BouncyCastleProvider());

        syncPluginsAndRefreshClassloader();

        // If setup is run in same process without exiting, we need reboot here!
        // Additionally ApplicationBooter also does boot
        LocalBooter.reboot(XldServerPaths.TYPE_DEFAULTS(), globalContextManager);

        if (!portIsAvailable()) {
            String name = deployMessageService.getMessage("xl.deploy.server");
            logger.error("Cannot start {}; port {} already in use.", name, port);
            logger.error("Perhaps another instance of {} is running.", name);
            System.exit(1);
        }

        if (logger.isDebugEnabled())
            logger.debug("Setting up Jetty server on port {} with context root {} using package {} for document root.",
                    port, webContextRoot, ServerConfiguration.DEFAULT_WEBCONTENT_PACKAGE);

        if (hasNonDefaultContextRoot()) {
            factory.setContextPath(webContextRoot);
        }

        factory.setPort(port);
        factory.setThreadPool(buildThreadPool());

        factory.addServerCustomizers(JettyServerConnector.create(serverConfiguration));
    }

    private void syncPluginsAndRefreshClassloader() {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        boolean syncSupportedByClassLoader = classLoader instanceof PluginClassLoader;
        boolean shouldSync = new SynchronizationCondition(jdbcTemplate, DEPLOYIT).shouldSync();

        if (shouldSync && syncSupportedByClassLoader) {
            PluginClassLoader cl = (PluginClassLoader) classLoader;

            cl.clearClasspathRoots();
            pluginSynchronizer.syncPlugins();
            cl.refreshDirs();
        }
    }

    private boolean hasNonDefaultContextRoot() {
        String webContextRoot = serverConfiguration.getWebContextRoot();
        if (webContextRoot == null || webContextRoot.isBlank()) {
            return false;
        }
        return !webContextRoot.equals(ServerConfiguration.DEFAULT_WEB_CONTEXT_ROOT);
    }

    private QueuedThreadPool buildThreadPool() {
        return new QueuedThreadPool(serverConfiguration.getMaxThreads(), serverConfiguration.getMinThreads());
    }

    private boolean portIsAvailable() {
        try {
            ServerSocket socket = new ServerSocket(serverConfiguration.getHttpPort());
            socket.close();
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    private static final Logger logger = LoggerFactory.getLogger(JettyServer.class);
}
