package com.xebialabs.xlplatform.rest.script.ui

import scala.xml.{NodeSeq, Node, Elem}
import com.xebialabs.xlplatform.rest.script.ResourceUtils

sealed trait IMenu extends Ordered[IMenu] {
  def weight: Int
  def label: String

  override def compare(that: IMenu): Int = {
    if (this.weight == that.weight) {
      this.label.compareTo(that.label)
    } else {
      this.weight - that.weight
    }
  }
}
case class MenuSeparator(weight: Int) extends IMenu {
  def label = "--------"
}

case class MenuItem(weight: Int, label: String, uri: String) extends IMenu
case class Menu(id: String, weight: Int, label: String, uri: Option[String], items: List[IMenu]) extends IMenu

object UiMenus {

  import ResourceUtils._

  def apply(filePattern: String): List[Menu] = {

    val uiPluginElements: List[Elem] = loadXmlResources(filePattern)

    def applyReferences(menus: List[IMenu], elems: NodeSeq): List[Menu] = {
      val lookup: Map[String, Menu] = menus.collect({
        case m: Menu => (m.id, m)
      }).toMap

      var notFoundMenuIds = List[String]()

      val resultedMenus = (elems \ "menu-ref").foldLeft(lookup)({
        case (l, n) =>
          val menuId: String = (n \ "@ref").text
          if (l.contains(menuId)) {
            val appendedMenu: Menu = appendItems(l(menuId), n)
            val appendedWithRefs = appendedMenu.copy(items = (appendedMenu.items.filterNot(_.isInstanceOf[Menu]) ++ applyReferences(appendedMenu.items, n)).sorted)
            l + ((menuId, appendedWithRefs))
          } else {
            notFoundMenuIds = menuId :: notFoundMenuIds
            l
          }
      }).values.toList

      if (!notFoundMenuIds.isEmpty) {
        throw new NotFoundException(s"Menus with the next IDs are not found: ${notFoundMenuIds mkString ", "}")
      }

      resultedMenus
    }

    def getMenuItem(e: Node): MenuItem = MenuItem((e \ "@weight").text.toInt, (e \ "@label").text, (e \ "@uri").text)
    def getMenu(e: Node): Menu = appendItems(Menu((e \ "@id").text, (e \ "@weight").text.toInt, (e \ "@label").text, (e \ "@uri").headOption.map(_.text), List()), e)

    def getMenuSeparator(e: Node): MenuSeparator = MenuSeparator((e \ "@weight").text.toInt)

    def appendItems(m: Menu, e: Node): Menu = m.copy(items = (m.items ++ e.child.collect({
      case node: Elem if node.label == "menu-item" => getMenuItem(node)
      case node: Elem if node.label == "menu" => getMenu(node)
      case node: Elem if node.label == "menu-separator" => getMenuSeparator(node)
    })).sorted)


    def checkMenuIdsOnDuplication(uiPluginElements: List[Elem]): Unit = {
      val allLevelMenuIds: List[String] = uiPluginElements.flatMap(e => e \\ "menu").map(p => (p \ "@id").text)

      def count[A](xs: List[A]): List[(A, Int)] = xs.distinct.map(x => (x, xs.count(_ == x)))
      val duplicatedIds: List[(String, Int)] = count(allLevelMenuIds).filter(p => p._2 > 1)

      if (!duplicatedIds.isEmpty)
        throw new DuplicationException(s"Duplicated IDs are not allowed: ${duplicatedIds.map(_._1) mkString ", "}")
    }


    val menus: List[IMenu] = uiPluginElements.flatMap(e => e \ "menu").map(getMenu)
    checkMenuIdsOnDuplication(uiPluginElements)

    applyReferences(menus, uiPluginElements)
  }


}
