package com.xebialabs.xlplatform.rest.script.endpoints

import spray.routing.{PathMatcher0, PathMatchers}
import org.slf4j.{LoggerFactory, Logger}
import com.xebialabs.xlplatform.rest.script.ResourceUtils
import java.net.URL
import grizzled.slf4j.Logging
import scala.xml.{Elem, Node}

trait EndpointDefinition {
  def path: String

  def getPathMatcher: PathMatcher0 = PathMatchers.separateOnSlashes(cleanPath)

  def cleanPath: String = {
    path.split('/').filterNot(_.isEmpty).mkString("/")
  }
}
case class ScalaEndpoint(path: String, traitClass: String, pfName: String) extends EndpointDefinition
case class ScriptEndpoint(path: String, method: String, script: URL) extends EndpointDefinition

object ScalaEndpoint extends Logging {
  def apply(path: String, elem: Elem): ScalaEndpoint = {
    val route: String = (elem \ "@route").text
    debug(s"Found configured extension route: $route")
    val lastDotIdx: Int = route.lastIndexOf('.')
    val (c, dot_f) = route.splitAt(lastDotIdx)
    ScalaEndpoint(path, c, dot_f.substring(1))
  }
}

object ScriptEndpoint extends Logging {

  def apply(path: String, elem: Elem): ScriptEndpoint = {
    ScriptEndpoint(path, (elem \ "@method").text, findFile((elem \ "@script").text))
  }

  private def findFile(p: String): URL = {
    Option(Thread.currentThread().getContextClassLoader.getResource(p)) match {
      case Some(url) => url
      case None => throw new IllegalArgumentException(s"Can not find script [$p]")
    }
  }
}

object ScriptEndpoints extends Logging {
  import ResourceUtils._
  def apply(config: String): Seq[ScriptEndpoint] = {
    (loadXmlResources(config) \ "endpoint").collect({case e: Elem if e.attribute("script").isDefined =>
      val path: String = (e \ "@path").text
      debug(s"Found dynamic script route mounted on path [$path]")
      ScriptEndpoint(path, e)
    })
  }
}


object ScalaEndpoints extends Logging {
  import ResourceUtils._
  def apply(config: String): Seq[ScalaEndpoint] = {
    (loadXmlResources(config) \ "route").collect({case e: Elem if e.attribute("route").isDefined =>
      val path: String = (e \ "@path").text
      debug(s"Found dynamic route mounted on path [$path]")
      ScalaEndpoint(path, e)
    })
  }
}
