package com.xebialabs.deployit.task.archive

import java.util.regex.Pattern
import javax.jcr._
import javax.jcr.query.QueryResult

import com.google.common.collect.Sets.newHashSet
import com.xebialabs.deployit.engine.api.execution._
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.jcr.JcrConstants._
import com.xebialabs.deployit.jcr.JcrUtils._
import com.xebialabs.deployit.jcr.grouping.GroupBy
import com.xebialabs.deployit.jcr.{JcrCallback, JcrTemplate}
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters
import com.xebialabs.deployit.task.TaskMetadata._
import com.xebialabs.deployit.task.TaskType.{CONTROL, INSPECTION}
import com.xebialabs.deployit.task.archive.JcrArchivedTaskSearchQueryBuilder.TASK_SELECTOR_NAME
import org.apache.jackrabbit.commons.JcrUtils
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.convert.wrapAll._
import scala.collection.convert.decorateAsJava._
import scala.collection.mutable.ListBuffer

@Component
class JcrTaskArchive extends Archive {

  @beans.BeanProperty
  @Autowired var jcrTemplate: JcrTemplate = _

  def this(jcrTemplate: JcrTemplate) = {
    this()
    this.jcrTemplate = jcrTemplate
  }

  import com.xebialabs.deployit.task.archive.JcrTaskArchive.{EXCLUDE_SPECIAL_ENV_TASK_NAMES, logger, qualifiedTaskIdPattern}

  @Override
  override def archive(task: TaskWithBlock) {
    logger.info("Archiving task [{}] with metadata [{}]", task.getId, task.getMetadata, "")

    jcrTemplate.execute { session: Session =>
      val rootNode = getTasksRootNode(session)
      val archiver = TaskArchiver(task)
      archiver.save(rootNode)
      session.save()
    }
  }

  def getTask(taskId: String, loadSteps: Boolean): ArchivedTask = {
    val matcher = qualifiedTaskIdPattern.matcher(taskId)
    if (matcher.matches()) {
      val internalId = EnvironmentId.encode(matcher.group(1)) + "/" + matcher.group(2) + "/" + matcher.group(3) + "/" + matcher.group(4)
      getTaskUsingFullyQualifiedPath(internalId, loadSteps)
    } else {
      getTaskUsingUuid(taskId, loadSteps)
    }
  }

  def purgeTask(taskUuid: String): Unit = {
    val params = new ArchivedTaskSearchParameters().withUniqueId(taskUuid)

    jcrTemplate.execute { session: Session =>
      val queryResult = executeJcrQuery(session, params)
      withTaskNode(queryResult) { taskNode =>
        if (taskNode.hasProperty(NodeNames.PACKAGE_DEPENDENCY_NAMES) &&
          taskNode.getProperty(NodeNames.PACKAGE_DEPENDENCY_NAMES).getValues.nonEmpty) {
          purgeSubTasks(taskUuid, session)
        }
        taskNode.remove()
      }

      session.save()
    }
  }

  private def purgeSubTasks(taskUuid: String, session: Session): Unit = {
    val params = new ArchivedTaskSearchParameters().childTasksFor(taskUuid)
    val queryResult = executeJcrQuery(session, params)
    withTaskNode(queryResult) { taskNode =>
      taskNode.remove()
    }
  }

  def searchTasks(params: ArchivedTaskSearchParameters): java.util.Collection[ArchivedTask] = {
    searchTasks(params, loadSteps = true)
  }

  def searchTasksWithoutLoadingSteps(params: ArchivedTaskSearchParameters): java.util.Collection[ArchivedTask] = {
    searchTasks(params, loadSteps = false)
  }

  def searchTasks(params: ArchivedTaskSearchParameters, callback: TaskCallback) {
    searchTasks(params, callback, loadSteps = true)
  }

  def searchTasksWithoutLoadingSteps(params: ArchivedTaskSearchParameters, callback: TaskCallback) {
    searchTasks(params, callback, loadSteps = false)
  }

  def getCloudEnvironmentTemplates: java.util.List[String] = {
    jcrTemplate.execute { session: Session =>
      val p = new ArchivedTaskSearchParameters().forAllCloudEnvironmentTemplates()
      val result = executeJcrQuery(session, p)

      JcrUtils
        .getNodes(result)
        .toList
        .map(node => ConfigurationId.decode(getProperty(node, CLOUD_ENVIRONMENT_TEMPLATE_ID)))
        .distinct
        .sorted
        .asJava
    }
  }

  def getAllTaskIds: java.util.List[String] = {
    jcrTemplate.execute { session: Session =>
      val queryResult = executeJcrQuery(session, new ArchivedTaskSearchParameters())
      getTaskIds(queryResult)
    }
  }

  def getAllEnvironments: java.util.List[String] = {
    jcrTemplate.execute { session: Session =>
      val envs = ListBuffer.empty[String]
      val nodes = session.getNode(TASKS_NODE_ID).getNodes
      while (nodes.hasNext) {
        val envId = EnvironmentId.decode(nodes.nextNode().getName)
        if (!EXCLUDE_SPECIAL_ENV_TASK_NAMES.contains(envId)) {
          envs += envId
        }
      }
      envs
    }
  }

  def searchTasksWithoutLoadingSteps(params: ArchivedTaskSearchParameters, groupBy: GroupBy): java.util.Collection[java.util.Map[String, Object]] = {
    jcrTemplate.execute { session: Session =>
      val iterator = executeJcrQuery(session, params).getRows
      while (iterator.hasNext) {
        val taskNode = iterator.nextRow().getNode(TASK_SELECTOR_NAME)
        groupBy.process(TaskReader(taskNode).readWithoutStep())
      }
      groupBy.getResult
    }
  }

  private def getTaskUsingFullyQualifiedPath(fullyQualifiedPath: String, loadSteps: Boolean): ArchivedTask = {
    try {
      jcrTemplate.execute { session: Session =>
        val tasksNode = getTasksRootNode(session)
        val taskNode = tasksNode.getNode(fullyQualifiedPath)
        val task = TaskReader(taskNode).readWithStep(loadSteps)

        if (task == null) {
          throw new NotFoundException("Cannot load task " + fullyQualifiedPath + " because that object with that id is not a task")
        }
        task
      }
    } catch {
      case exc: PathNotFoundException =>
        throw new NotFoundException("Cannot load task " + fullyQualifiedPath + " because it does not exist")
    }
  }

  private def getTaskUsingUuid(taskUuid: String, loadSteps: Boolean): ArchivedTask = {
    val taskSearch = new ArchivedTaskSearchParameters()
    taskSearch.withUniqueId(taskUuid)

    val deploymentTaskInfo = if (loadSteps) {
      searchTasks(taskSearch)
    } else {
      searchTasksWithoutLoadingSteps(taskSearch)
    }
    if (deploymentTaskInfo.size() == 1) {
      deploymentTaskInfo.iterator().next()
    } else if (deploymentTaskInfo.isEmpty) {
      throw new TaskNotFoundException("archive", taskUuid)
    } else {
      throw new NotFoundException("Cannot load task " + taskUuid + " because there are multiple tasks with same id. " + deploymentTaskInfo)
    }
  }

  private def getTasksRootNode(session: Session): Node = {
    session.getRootNode.getNode(TASKS_NODE_NAME)
  }

  private def searchTasks(params: ArchivedTaskSearchParameters, loadSteps: Boolean): Seq[ArchivedTask] = {
    jcrTemplate.execute { session: Session =>
      val queryResult = executeJcrQuery(session, params)
      val listBuffer = ListBuffer.empty[ArchivedTask]

      mapQueryResultToTasks(queryResult, loadSteps) { (task: ArchivedTask) => listBuffer += task}

      listBuffer
    }
  }

  private def searchTasks(params: ArchivedTaskSearchParameters, callback: TaskCallback, loadSteps: Boolean) {
    jcrTemplate.execute { session: Session =>
      val queryResult = executeJcrQuery(session, params)
      mapQueryResultToTasks(queryResult, loadSteps)(callback)
    }
  }

  private def executeJcrQuery(session: Session, params: ArchivedTaskSearchParameters): QueryResult = {
    val qm = session.getWorkspace.getQueryManager
    val queryBuilder = new JcrArchivedTaskSearchQueryBuilder(qm, session.getValueFactory, params)
    val query = queryBuilder.buildQuery()
    query.execute()
  }

  private def mapQueryResultToTasks(result: QueryResult, loadSteps: Boolean)(callback: TaskCallback) {
    withTaskNode(result) { taskNode =>
      val task = TaskReader(taskNode).readWithStep(loadSteps)
      callback.doWithTask(task)
    }
  }

  private def withTaskNode(result: QueryResult)(f: Node => Unit): Unit = {
    val iterator = result.getRows
    while (iterator.hasNext) {
      try {
        val taskNode = iterator.nextRow().getNode(TASK_SELECTOR_NAME)
        f(taskNode)
      } catch {
        case rre: RepositoryException =>
          logger.error("Error while reading archived task", rre)
      }
    }
  }

  private def getTaskIds(queryResult: QueryResult): java.util.List[String] = {
    val taskIds = ListBuffer.empty[String]
    val iterator = queryResult.getRows
    while (iterator.hasNext) {
      try {
        taskIds.add(getProperty(iterator.nextRow().getNode(TASK_SELECTOR_NAME), TASK_ID_PROPERTY_NAME))
      } catch {
        case rre: RepositoryException =>
          logger.error("Error while reading archived task", rre)
      }
    }
    taskIds
  }

  implicit def jcrTemplateWrapper[T](f: (Session => T)): JcrCallback[T] = {
    new JcrCallback[T] {
      override def doInJcr(session: Session): T = f(session)
    }
  }

  implicit def taskCallbackWrapper[T](f: (ArchivedTask => _)): TaskCallback = {
    new TaskCallback {
      override def doWithTask(task: ArchivedTask): Unit = f.apply(task)
    }
  }

}

object JcrTaskArchive {
  // properties of the task node
  private val qualifiedTaskIdPattern = Pattern.compile("^(.*)/(.*)/(.*)/(.*)$")
  // ie. env/app/version/uuid

  import com.xebialabs.deployit.task.archive.EnvironmentId.ENVIRONMENTS_ROOT

  private val EXCLUDE_SPECIAL_ENV_TASK_NAMES = newHashSet(ENVIRONMENTS_ROOT + INSPECTION, ENVIRONMENTS_ROOT + CONTROL, ENVIRONMENTS_ROOT + "rep:policy")

  private val logger = LoggerFactory.getLogger(classOf[JcrTaskArchive])


}