package com.xebialabs.deployit.task.archive;

import com.xebialabs.deployit.engine.api.execution.*;
import org.joda.time.DateTime;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class ArchivedTask implements TaskWithSteps, TaskWithBlock {

    protected List<StepState> lazySteps = null;
    private int lazyNrSteps = -1;

    private TaskExecutionState state;
    private String id;
    private String description;
    private DateTime startDate;
    private DateTime completionDate;
    private Map<String, String> metadata = newHashMap();
    private int failureCount;
    private String owner;
    private BlockState block;
    private List<String> activeBlocks = newArrayList();
    private DateTime scheduledDate;
    private List<TaskPackageDependency> packageDependencies = newArrayList();

    @Override
    public List<StepState> getSteps() {
        return lazyGetSteps();
    }

    @Override
    public StepState getStep(int stepNr) {
        checkArgument(stepNr > 0 && stepNr <= getNrSteps(), "Not a valid step number [%s]", stepNr);
        return lazyGetSteps().get(stepNr - 1);
    }

    @Override
    public TaskExecutionState getState() {
        return state;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public DateTime getScheduledDate() {
        return scheduledDate;
    }

    public void setScheduledDate(final DateTime scheduledDate) {
        this.scheduledDate = scheduledDate;
    }

    @Override
    public DateTime getStartDate() {
        return startDate;
    }

    @Override
    public DateTime getCompletionDate() {
        return completionDate;
    }

    @Override
    public int getNrSteps() {
        if (lazyNrSteps == -1) {
            lazyNrSteps = lazyGetSteps().size();
        }
        return lazyNrSteps;
    }

    /**
     * @deprecated Use {@link #getCurrentStepNrs()} instead.
     */
    @Deprecated
    @Override
    public int getCurrentStepNr() {
        return 0;
    }

    @Override
    public List<Integer> getCurrentStepNrs() {
        return newArrayList(0);
    }

    @Override
    public Map<String, String> getMetadata() {
        return metadata;
    }

    @Override
    public int getFailureCount() {
        return failureCount;
    }

    @Override
    public String getOwner() {
        return owner;
    }

    @Override
    public BlockState getBlock() {
        return block;
    }

    @Override
    public List<String> getActiveBlocks() {
        return activeBlocks;
    }

    public void setBlock(final BlockState block) {
        this.block = block;
    }

    public void setState(TaskExecutionState state) {
        this.state = state;
    }

    public void setId(String id) {
        this.id = id;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public void setStartDate(DateTime startDate) {
        this.startDate = startDate;
    }

    public void setCompletionDate(DateTime completionDate) {
        this.completionDate = completionDate;
    }

    public void setMetadata(Map<String, String> metadata) {
        this.metadata = metadata;
    }

    public void setFailureCount(int failureCount) {
        this.failureCount = failureCount;
    }

    public void setOwner(String owner) {
        this.owner = owner;
    }

    protected List<StepState> lazyGetSteps() {
        if (lazySteps == null) {
            if (block == null) {
                lazySteps = newArrayList();
            } else {
                lazySteps = getStepsFromBlock(block);
            }
        }
        return lazySteps;
    }

    private List<StepState> getStepsFromBlock(final BlockState block) {
        if (block instanceof ArchivedStepBlock) {
            ArchivedStepBlock stepBlock = (ArchivedStepBlock) block;
            return stepBlock.getSteps();

        } else if (block instanceof CompositeBlockState) {
            CompositeBlockState compBlock = (CompositeBlockState) block;
            List<StepState> steps = newArrayList();
            for (BlockState b : compBlock.getBlocks()) {
                steps.addAll(getStepsFromBlock(b));
            }
            return steps;

        } else if (block instanceof ArchivedPhase) {
            ArchivedPhase phase = (ArchivedPhase) block;
            return getStepsFromBlock(phase.underlyingBlock);

        } else if (block instanceof ArchivedPhaseContainer) {
            ArchivedPhaseContainer container = (ArchivedPhaseContainer) block;
            List<StepState> steps = newArrayList();
            for (BlockState phase : container.phases) {
                steps.addAll(getStepsFromBlock(phase));
            }
            return steps;
        }
        return new ArrayList<>();
    }

    @Override
    public List<TaskPackageDependency> getPackageDependencies() {
        return packageDependencies;
    }
}
