package com.xebialabs.deployit.task;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.joda.time.DateTime;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

public class ArchivedTaskSearchParameters {

    public enum DateRange {
        AFTER, BEFORE, BETWEEN, NONE
    }

    public enum Status {
        COMPLETED, CANCELLED, COMPLETED_AFTER_RETRY, ANY
    }

    protected DateTime startDate;
    protected DateTime endDate;
    protected String executedBy;
    protected Set<Application> applications = newHashSet();
    protected Set<String> environments = newHashSet();
    protected Set<String> cloudEnvironmentTemplates = newHashSet();
    protected boolean allCloudEnvironmentTemplates;
    protected String taskUuid;
    protected String parentTaskUuid;
    protected DateRange dateRangeSearch = ArchivedTaskSearchParameters.DateRange.NONE;
    protected Status status = ArchivedTaskSearchParameters.Status.ANY;
    protected EnumSet<TaskType> taskTypes = EnumSet.noneOf(TaskType.class);
    protected String orderBy;
    protected boolean includeMainIfDependencyMatches;
    protected boolean includeDependencies;

    public ArchivedTaskSearchParameters() {
    }

    public ArchivedTaskSearchParameters(ArchivedTaskSearchParameters p) {
        this.startDate = p.startDate;
        this.endDate = p.endDate;
        this.executedBy = p.executedBy;
        this.applications = newHashSet(p.applications);
        this.environments = newHashSet(p.environments);
        this.cloudEnvironmentTemplates = newHashSet(p.cloudEnvironmentTemplates);
        this.allCloudEnvironmentTemplates = p.allCloudEnvironmentTemplates;
        this.dateRangeSearch = p.dateRangeSearch;
        this.status = p.status;
        this.taskUuid = p.taskUuid;
        this.parentTaskUuid = p.parentTaskUuid;
        this.taskTypes = p.taskTypes;
        this.orderBy = p.orderBy;
        this.includeDependencies = p.includeDependencies;
        this.includeMainIfDependencyMatches = p.includeMainIfDependencyMatches;
    }

    public ArchivedTaskSearchParameters orderBy(String property) {
        this.orderBy = property;
        return this;
    }

    public ArchivedTaskSearchParameters executedByUser(String user) {
        this.executedBy = user;
        return this;
    }

    public ArchivedTaskSearchParameters createdBetween(DateTime startDate, DateTime endDate) {
        checkNotNull(startDate, "Start date cannot be null");
        checkNotNull(endDate, "End date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.BETWEEN;
        if (startDate.isAfter(endDate)) {
            this.startDate = endDate;
            this.endDate = startDate;
        } else {
            this.startDate = startDate;
            this.endDate = endDate;
        }
        return this;
    }

    public ArchivedTaskSearchParameters createdOnOrBefore(DateTime endDate) {
        checkNotNull(endDate, "End date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.BEFORE;
        this.endDate = endDate;
        return this;
    }

    public ArchivedTaskSearchParameters createdOnOrAfter(DateTime startDate) {
        checkNotNull(startDate, "Start date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.AFTER;
        this.startDate = startDate;
        return this;
    }

    /**
     * Sets the datetime search constraints to BEFORE, AFTER, BETWEEN or not at all, depending on
     * whether <tt>startDate</tt> or <tt>endDate</tt> (or both) are null
     * @param startDate the datetime range start; can be null
     * @param endDate the datetime range end; can be null
     * @return {@code this}
     */
    public ArchivedTaskSearchParameters inDateTimeRange(DateTime startDate, DateTime endDate) {
        if (startDate == null && endDate != null) {
            return this.createdOnOrBefore(endDate);
        } else if (startDate != null && endDate == null) {
            return this.createdOnOrAfter(startDate);
        } else if (startDate != null & endDate != null) {
            return this.createdBetween(startDate, endDate);
        } else {
            return this;
        }
    }

    public ArchivedTaskSearchParameters forApplicationAndVersion(String application, String version) {
        Checks.checkTrue(environments.isEmpty(), "Can't query for both environments and applications.");
        Checks.checkTrue(cloudEnvironmentTemplates.isEmpty(), "Can't query for both applications and cloudEnvironmentTemplates.");

        applications.add(new Application(application, version));

        return this;
    }

    public ArchivedTaskSearchParameters forApplication(String application) {
        return forApplicationAndVersion(application, null);
    }

    public ArchivedTaskSearchParameters thatAreOfType(EnumSet<TaskType> types) {
        this.taskTypes = types;
        return this;
    }

    public ArchivedTaskSearchParameters forEnvironment(String environment) {
        Checks.checkTrue(applications.isEmpty(), "Can't query for both environments and applications.");
        Checks.checkTrue(cloudEnvironmentTemplates.isEmpty(), "Can't query for both environments and cloudEnvironmentTemplates.");

        environments.add(environment);

        return this;
    }

    public ArchivedTaskSearchParameters forCloudEnvironmentTemplate(String cloudEnvironmentTemplate) {
        Checks.checkTrue(applications.isEmpty(), "Can't query for both cloudEnvironmentTemplates and applications.");
        Checks.checkTrue(environments.isEmpty(), "Can't query for both cloudEnvironmentTemplates and environments.");

        cloudEnvironmentTemplates.add(cloudEnvironmentTemplate);

        return this;
    }

    public ArchivedTaskSearchParameters forAllCloudEnvironmentTemplates() {
        Checks.checkTrue(applications.isEmpty(), "Can't query for all cloudEnvironmentTemplates and applications.");
        Checks.checkTrue(environments.isEmpty(), "Can't query for all cloudEnvironmentTemplates and environments.");
        Checks.checkTrue(cloudEnvironmentTemplates.isEmpty(), "Can't query for all cloudEnvironmentTemplates and specific cloudEnvironmentTemplate(s).");

        allCloudEnvironmentTemplates = true;

        return this;
    }

    public ArchivedTaskSearchParameters forFilterType(FilterType filterType, List<ConfigurationItemId> cis) {
        for (ConfigurationItemId appOrEnvName : cis) {
            forFilterType(filterType, appOrEnvName);
        }

        return this;
    }

    public ArchivedTaskSearchParameters forFilterType(FilterType filterType, ConfigurationItemId appOrEnvName) {
        switch (filterType) {
        case APPLICATION:
            forApplication(appOrEnvName.getId());
            break;
        case ENVIRONMENT:
            forEnvironment(appOrEnvName.getId());
            break;
        case CLOUD_ENVIRONMENT_TEMPLATE:
            forCloudEnvironmentTemplate(appOrEnvName.getId());
            break;
        default:
            break;
        }
        return this;
    }

    public ArchivedTaskSearchParameters thatWasCancelled() {
        status = Status.CANCELLED;
        return this;
    }

    public ArchivedTaskSearchParameters thatCompleted() {
        status = Status.COMPLETED;
        return this;
    }

    public ArchivedTaskSearchParameters thatCompletedAfterRetry() {
        status = Status.COMPLETED_AFTER_RETRY;
        return this;
    }

    public ArchivedTaskSearchParameters withUniqueId(String uuid) {
        this.taskUuid = uuid;
        return this;
    }

    public ArchivedTaskSearchParameters includeDependencies() {
        this.includeDependencies = true;
        return this;
    }

    public ArchivedTaskSearchParameters includeMainIfDependencyMatches() {
        this.includeMainIfDependencyMatches = true;
        return this;
    }

    public ArchivedTaskSearchParameters childTasksFor(String uuid) {
        this.parentTaskUuid = uuid;
        this.includeDependencies = true;
        this.includeMainIfDependencyMatches = false;
        return this;
    }

    protected static class Application {
        public final String name;
        public final String version;

        Application(String name, String version) {
            this.name = name;
            this.version = version;
        }
    }
}
