package com.xebialabs.deployit.service.dependency

import com.xebialabs.deployit.plugin.api.deployment.specification.Operation
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation.{DESTROY, MODIFY}
import com.xebialabs.deployit.plugin.api.semver.VersionRange
import com.xebialabs.deployit.plugin.api.udm.{DeployedApplication => UdmDeployedApplication, DeploymentPackage => UdmPackage, Version => UdmVersion}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.DependencyConstraints.DependencyException
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.convert.wrapAll._
import scala.collection.mutable

@Component
class DependencyService @Autowired()(dependencyFinder: DependencyFinder, val repositoryService: RepositoryService) extends RepositoryServiceAware {

  def resolveMultiDeployment(envId: String, udmVersion: UdmVersion): java.util.List[DeploymentResult] = {
    resolveMultiDeployment(envId, udmVersion, findDependencies(udmVersion))
  }

  def validateDependenciesNotBrokenOnUndeploy(envId: String, toUnDeployedPackage: UdmPackage): Unit = {
    val deps = allDependencies(loadDeployedApplications(envId))
    validateDependenciesNotBroken(toUnDeployedPackage, deps, DESTROY)(_.isEmpty)
  }

  private[dependency] def resolveMultiDeployment(envId: String, main: UdmVersion, dependencies: List[UdmVersion]): List[DeploymentResult] = {
    val allDeployedApplications: Map[String, UdmDeployedApplication] = loadDeployedApplications(envId).map(da => getApplicationIdFromVersion(da.getVersion) -> da).toMap
    val deps: mutable.Map[String, Map[String, String]] = mutable.Map(allDependencies(allDeployedApplications.values.toList).toSeq: _*)

    def validate(udmVersion: UdmVersion): Unit = validateDependenciesNotBroken(udmVersion, deps.toMap, MODIFY) {
      _.map(r => new VersionRange(r))
        .map(vr => vr.includes(parseVersion(udmVersion.getVersion)))
        .getOrElse(true)
    }

    def isMain(udmVersion: UdmVersion) = udmVersion == main

    val results = (main +: dependencies).map { pck =>
      validate(pck)
      if (deps.contains(pck.getApplication.getId)) {
        deps.put(pck.getApplication.getId, extractDeps(pck))
      }
      DeploymentResult(pck, allDeployedApplications.get(pck.getApplication.getId), isMain(pck))
    }

    results.filterNot {
      case DeploymentResult(pck, Some(app), isMain) => pck == app.getVersion && !isMain
      case _ => false
    }.reverse
  }

  private[this] def extractDeps(v: UdmVersion): Map[String, String] = v match {
    case udmVer: UdmPackage => {
      val verCi: UdmPackage = repositoryService.read(udmVer.getId, 1)
      verCi.getApplicationDependencies.toMap
    }
    case _ => Map()
  }

  private[this] def allDependencies(deployedApplications: List[UdmDeployedApplication]): Map[String, Map[String, String]] = {
    deployedApplications.map(da => getApplicationIdFromVersion(da.getVersion) -> extractDeps(da.getVersion)).toMap
  }

  private[this] def getApplicationIdFromVersion(version: UdmVersion) = version.getId.substring(0, version.getId.lastIndexOf('/'))

  private[this] def validateDependenciesNotBroken(udmVersion: UdmVersion, dependencies: Map[String, Map[String, String]], op: Operation)(isValid: Option[String] => Boolean) {
    dependencies.foreach { case (appId, deps) =>
      if (!isValid(deps.get(udmVersion.getApplication.getName))) {
        val range = deps(udmVersion.getApplication.getName)
        throw new DependencyException( s"""Application "${udmVersion.getApplication.getId}" cannot be ${opToVerb(op)}, because the deployed application "$appId" depends on its current version. The required version range is '$range'.""")
      }
    }
  }

  private[this] def opToVerb(op: Operation): String = op match {
    case MODIFY => "upgraded"
    case DESTROY => "undeployed"
    case _ => "deployed"
  }

  private[this] def loadDeployedApplications(envId: String): List[UdmDeployedApplication] = {
    repositoryService.listEntities[UdmDeployedApplication](typedSearchParameters[UdmDeployedApplication].setParent(envId).setDepth(1)).toList
  }

  private[this] def findDependencies(udmVersion: UdmVersion): List[UdmVersion] = udmVersion match {
    case pck: UdmPackage =>
      val dependencies = dependencyFinder.find(pck)
      DependencyConstraints.validateOutboundDependencies(dependencies)
      dependencies.allRequiredVersionsInTopologicalOrder
    case _ => Nil
  }
}
