package com.xebialabs.deployit.service.dependency

import com.xebialabs.deployit.plugin.api.semver.VersionRange
import com.xebialabs.deployit.plugin.api.udm.{Application => UdmApplication, DeploymentPackage => UdmVersion}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.DependencyConstraints.DependencyException
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.language.implicitConversions
import scala.util.Try
import com.github.zafarkhaja.semver.{Version => OsgiVersion}

@Component
class DependencyFinder @Autowired() (val repositoryService: RepositoryService) extends RepositoryServiceAware with Logging {

  import collection.convert.wrapAsScala._

  case class Dependency(range : String, udmApplication: Option[UdmApplication], udmVersion: Option[UdmVersion])

  def find(deploymentPackage: UdmVersion): OutboundDependencies = {
    implicit val allDependencies = graph()
    implicit val addedNodes = mutable.MutableList[String]()
    findDependencies(
      Application.main(deploymentPackage.getApplication.getId, deploymentPackage.getVersion),
      deploymentPackage.getApplicationDependencies.toMap
    )
    allDependencies
  }

  private[this] def findDependencies(app: Application, dependencies: Map[String, String])
                              (implicit applicationDependencies: OutboundDependencies, addedNodes: mutable.MutableList[String]) {
    if (!addedNodes.contains(app.id)) {
      val resolved = resolveDependencies(dependencies, app)
      addedNodes += app.id
      applicationDependencies.add(app)
      findTransientDependencies(resolved)
      findDirectDependencies(app, resolved)
    }
  }

  private[this] def findDirectDependencies(app: Application, resolvedDependencies: Map[String, Dependency])
                                     (implicit applicationDependencies: OutboundDependencies): Unit = {
    import scalax.collection.edge.Implicits._
    resolvedDependencies.foreach { case (appName, Dependency(range, udmApp, udmVersion)) =>
      applicationDependencies.add((app ~+> node(appName, udmApp))(Label(udmVersion, range)))
    }
  }

  private[this] def findTransientDependencies(resolvedDependencies: Map[String, Dependency])
                                        (implicit applicationDependencies: OutboundDependencies, addedNodes: mutable.MutableList[String]): Unit = {
    resolvedDependencies.collect {
      case (appName, Dependency(_, a@Some(udmApp), Some(udmVersion))) =>
        findDependencies(Application.resolved(udmApp.getId), udmVersion.getApplicationDependencies.toMap)
    }
  }

  private[this] def resolveDependencies(dependencies: Map[String, String], dependingApplication: Application): Map[String, Dependency] = dependencies.map {
    case (appName, range) =>
      val app = loadApplication(appName, dependingApplication)
      val version = app.flatMap(a => findMaxVersion(a.getId, new VersionRange(range)))
      appName -> Dependency(range, app, version)
  }

  private[this] def loadApplication(name: String, dependingApplication: Application): Option[UdmApplication] = {
    val applications = repositoryService.listEntities[UdmApplication](typedSearchParameters[UdmApplication].setName(name))
    if (applications.size() > 1)
      throw new DependencyException(s"""Error while trying to resolve the dependencies of application "$dependingApplication". Multiple applications found with the name "$name". Please rename one of your applications, application names should be unique.""")
    applications.toSeq.headOption
  }

  private[this] def findMaxVersion(appId: String, range: VersionRange): Option[UdmVersion] = {

    val udmVersions = repositoryService.listEntities[UdmVersion](typedSearchParameters[UdmVersion].setParent(appId))
    val resolvedVersions = udmVersions.toSeq.map(uv => uv -> Try(uv.toOsgi).toOption).toMap
    val maxOsgi = resolvedVersions.values.flatten.foldLeft(Option.empty[OsgiVersion])((result, v) => result max v.in(range))
    resolvedVersions.view.filter(_._2.isDefined).find {
      _._2 == maxOsgi
    }.map(_._1)
  }

  private[this] def node(appName: String, app: Option[UdmApplication]) = app.map(a => Application.resolved(a.getId)).getOrElse(Application.unresolved(appName))
}
