package com.xebialabs.deployit.task.archive;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.TaskMetadata;
import com.xebialabs.deployit.task.TaskType;
import org.joda.time.DateTime;

import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.query.QueryManager;
import javax.jcr.query.qom.*;
import java.util.List;

import static com.xebialabs.deployit.jcr.JcrConstants.TASK_NODETYPE_NAME;
import static javax.jcr.query.qom.QueryObjectModelConstants.*;
import static com.xebialabs.deployit.task.archive.NodeNames.*;

public class JcrArchivedTaskSearchQueryBuilder extends ArchivedTaskSearchParameters {

    private final ValueFactory values;
    private final QueryObjectModelFactory queries;

    static final String TASK_SELECTOR_NAME = "task";

    public JcrArchivedTaskSearchQueryBuilder(QueryManager qm, ValueFactory vf, ArchivedTaskSearchParameters p) {
        super(p);

        this.values = vf;
        this.queries = qm.getQOMFactory();
    }

    public QueryObjectModel buildQuery() throws RepositoryException {
        List<Constraint> constraints = Lists.newArrayList();

        appendTaskUuidCriteria(constraints);
        appendTaskTypeCriteria(constraints);
        appendEnvironmentCriteria(constraints);
        appendApplicationCriteria(constraints);
        appendCloudEnvironmentTemplateCriteria(constraints);
        appendAllCloudEnvironmentTemplatesCriteria(constraints);
        appendExecutedByCriteria(constraints);
        appendStatusCriteria(constraints);
        appendDateRangeCriteria(constraints);
        appendTaskDependenciesCriteria(constraints);
        appendChildTaskByUuidCriteria(constraints);

        Ordering[] ordering = createOrderBy();

        Selector selector = queries.selector(TASK_NODETYPE_NAME, TASK_SELECTOR_NAME);

        if (constraints.isEmpty()) {
            return queries.createQuery(selector, null, ordering, null);
        } else {
            return queries.createQuery(selector, and(constraints), ordering, null);
        }
    }

    private Ordering[] createOrderBy() throws RepositoryException {
        if (this.orderBy != null) {
            return new Ordering[]{
                    queries.ascending(queries.propertyValue(TASK_SELECTOR_NAME, this.orderBy))};
        } else {
            return null;
        }
    }

    private void appendTaskUuidCriteria(List<Constraint> constraints) throws RepositoryException {
        if (isNotBlank(taskUuid)) {
            Literal literal = queries.literal(values.createValue(taskUuid));
            constraints.add(queries.comparison(queries.nodeName(TASK_SELECTOR_NAME), JCR_OPERATOR_EQUAL_TO, literal));
        }
    }

    private static boolean isNotBlank(String s) {
        return !Strings.nullToEmpty(s).trim().isEmpty();
    }

    private void appendTaskTypeCriteria(List<Constraint> constraints) throws RepositoryException {
        if (taskTypes.isEmpty()) {
            return;
        }

        List<Constraint> taskTypeConstraints = Lists.newArrayList();
        for (TaskType tt : taskTypes) {
            taskTypeConstraints.add(taskTypeConstraint(tt));
        }
        constraints.add(or(taskTypeConstraints));
    }

    private void appendEnvironmentCriteria(List<Constraint> constraints) throws RepositoryException {
        if (environments.isEmpty()) {
            constraints.add(queries.descendantNode(TASK_SELECTOR_NAME, JcrConstants.TASKS_NODE_ID));
            return;
        }

        List<Constraint> environmentConstraints = Lists.newArrayList();
        for (String environment : environments) {
            environmentConstraints.add(environmentConstraint(environment));
        }
        constraints.add(or(environmentConstraints));
    }

    private void appendCloudEnvironmentTemplateCriteria(List<Constraint> constraints) throws RepositoryException {
        if (cloudEnvironmentTemplates.isEmpty()) {
            return;
        }

        List<Constraint> cloudEnvironmentTemplateConstraints = Lists.newArrayList();
        for (String cet : cloudEnvironmentTemplates) {
            cloudEnvironmentTemplateConstraints.add(cloudEnvironmentTemplateConstraint(cet));
        }
        constraints.add(or(cloudEnvironmentTemplateConstraints));
    }

    private void appendAllCloudEnvironmentTemplatesCriteria(List<Constraint> constraints) throws RepositoryException {
        if (!allCloudEnvironmentTemplates) {
            return;
        }
        constraints.add(queries.propertyExistence(TASK_SELECTOR_NAME, CLOUD_ENVIRONMENT_TEMPLATE_ID));
    }

    private void appendApplicationCriteria(List<Constraint> constraints) throws RepositoryException {
        if (applications.isEmpty()) {
            return;
        }

        List<Constraint> applicationConstraints = Lists.newArrayList();
        for (Application application : applications) {
            applicationConstraints.add(applicationConstraint(application));
        }
        constraints.add(or(applicationConstraints));
    }

    private void appendExecutedByCriteria(List<Constraint> constraints) throws RepositoryException {
        if (isNotBlank(executedBy)) {
            constraints.add(equalConstraint(OWNING_USER, executedBy));
        }
    }

    private void appendStatusCriteria(List<Constraint> constraints) throws RepositoryException {
        switch (status) {
        case COMPLETED:
            constraints.add(queries.and(equalConstraint(STATE, TaskExecutionState.DONE.name()), equalConstraint(FAILURE_COUNT, 0l)));
            break;
        case CANCELLED:
            constraints.add(equalConstraint(STATE, TaskExecutionState.CANCELLED.name()));
            break;
        case COMPLETED_AFTER_RETRY:
            Constraint stateConstraint = equalConstraint(STATE, TaskExecutionState.DONE.name());
            Constraint failureConstraint = greaterThanConstraint(FAILURE_COUNT, 0l);
            constraints.add(queries.and(stateConstraint, failureConstraint));
            break;
        }
    }

    private void appendDateRangeCriteria(List<Constraint> constraints) throws RepositoryException {
        switch (dateRangeSearch) {
            case AFTER:
                DateTime startDateAsCal = cloneAndSetTimeToStartOfDay(startDate);
                constraints.add(greaterThanOrEqualConstraint(START_DATE, startDateAsCal));
                break;

            case BEFORE:
                DateTime endDateAsCal = cloneAndSetTimeToEndOfDay(endDate);
                constraints.add(lessThanOrEqualConstraint(START_DATE, endDateAsCal));
                break;

            case BETWEEN:
                DateTime betweenStartDate = cloneAndSetTimeToStartOfDay(startDate);
                DateTime betweenEndDate = cloneAndSetTimeToEndOfDay(endDate);
                Constraint startConstraint = greaterThanOrEqualConstraint(START_DATE, betweenStartDate);
                Constraint endConstraint = lessThanOrEqualConstraint(START_DATE, betweenEndDate);
                constraints.add(startConstraint);
                constraints.add(endConstraint);
        }
    }

    private void appendTaskDependenciesCriteria(List<Constraint> constraints) throws RepositoryException {
        if (!includeDependencies) {
            constraints.add(queries.not(queries.propertyExistence(TASK_SELECTOR_NAME, PARENT_TASK_ID)));
        }
    }

    private void appendChildTaskByUuidCriteria(List<Constraint> constraints) throws RepositoryException {
        if (parentTaskUuid != null) {
            Constraint existence = queries.propertyExistence(TASK_SELECTOR_NAME, PARENT_TASK_ID);
            Constraint parentUuid = addConstraint(PARENT_TASK_ID, JCR_OPERATOR_EQUAL_TO, values.createValue(parentTaskUuid));
            constraints.add(queries.and(existence, parentUuid));
        }
    }

    private Constraint equalConstraint(String propName, String value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_EQUAL_TO, values.createValue(value));
    }

    private Constraint equalConstraint(String propName, long value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_EQUAL_TO, values.createValue(value));
    }

    private Constraint greaterThanConstraint(String propName, long value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_GREATER_THAN, values.createValue(value));
    }

    private Constraint greaterThanOrEqualConstraint(String propName, DateTime value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_GREATER_THAN_OR_EQUAL_TO, values.createValue(value.toGregorianCalendar()));
    }

    private Constraint lessThanOrEqualConstraint(String propName, DateTime value) throws RepositoryException {
        return addConstraint(propName, JCR_OPERATOR_LESS_THAN_OR_EQUAL_TO, values.createValue(value.toGregorianCalendar()));
    }

    private Constraint addConstraint(String propName, String op, Value v) throws RepositoryException {
        return queries.comparison(queries.propertyValue(TASK_SELECTOR_NAME, propName), op, queries.literal(v));
    }

    private Constraint environmentConstraint(String environment) throws RepositoryException {
        String path = JcrConstants.TASKS_NODE_ID + "/" + EnvironmentId$.MODULE$.encode(environment);

        return queries.descendantNode(TASK_SELECTOR_NAME, path);
    }

    private Constraint cloudEnvironmentTemplateConstraint(String cloudEnvironmentTemplate) throws RepositoryException {
        return equalConstraint(TaskMetadata.CLOUD_ENVIRONMENT_TEMPLATE_ID, ConfigurationId$.MODULE$.encode(cloudEnvironmentTemplate));
    }

    private Constraint applicationConstraint(Application application) throws RepositoryException {
        Constraint constraint = equalConstraint(APPLICATION, application.name);

        if (application.version != null) {
            constraint = queries.and(constraint, equalConstraint(VERSION, application.version));
        }

        if (includeMainIfDependencyMatches) {
            Constraint depConstraint = equalConstraint(PACKAGE_DEPENDENCY_NAMES, application.name);
            if (application.version != null) {
                depConstraint = queries.and(depConstraint, equalConstraint(PACKAGE_DEPENDENCY_NAMES_AND_VERSIONS, application.name + "/" + application.version));
            }
            return queries.or(constraint, depConstraint);
        } else {
            return constraint;
        }
    }

    private Constraint taskTypeConstraint(TaskType taskType) throws RepositoryException {
        return equalConstraint(DEPLOYMENT_TYPE, taskType.toString());
    }

    private Constraint and(List<Constraint> constraints) throws RepositoryException {
        Checks.checkTrue(!constraints.isEmpty(), "List of constraints may not be empty to combine into AND constraint.");

        Constraint andConstraint = constraints.get(0);
        for (int i = 1; i < constraints.size(); i++) {
            andConstraint = queries.and(andConstraint, constraints.get(i));
        }
        return andConstraint;
    }

    private Constraint or(List<Constraint> constraints) throws RepositoryException {
        Checks.checkTrue(!constraints.isEmpty(), "List of constraints may not be empty to combine into OR constraint.");

        Constraint orConstraint = constraints.get(0);
        for (int i = 1; i < constraints.size(); i++) {
            orConstraint = queries.or(orConstraint, constraints.get(i));
        }
        return orConstraint;
    }

    private static DateTime cloneAndSetTimeToStartOfDay(DateTime date) {
        return date.withHourOfDay(0).withMinuteOfHour(0).withSecondOfMinute(0).withMillisOfSecond(0);
    }

    private static DateTime cloneAndSetTimeToEndOfDay(DateTime date) {
        return date.withHourOfDay(23).withMinuteOfHour(59).withSecondOfMinute(59).withMillisOfSecond(999);
    }
}
