package com.xebialabs.deployit.service.importer;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.io.Files;
import com.xebialabs.deployit.engine.spi.command.CreateCisCommand;
import com.xebialabs.deployit.engine.spi.command.RepositoryBaseCommand;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.xld.Distribution;
import com.xebialabs.deployit.plugin.api.xld.DistributionVersion;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.server.api.importer.*;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xlplatform.artifact.ArtifactEnricher;
import de.schlichtherle.truezip.file.TFile;
import nl.javadude.scannit.Scannit;
import nl.javadude.t2bus.event.strategy.ThrowingRuntimeExceptionHandlerStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.io.File;
import java.lang.reflect.Modifier;
import java.util.*;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.IMPORT_INITIAL;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.IMPORT_UPGRADE;

public class ImporterServiceImpl implements ImporterService {

    private RepositoryService repositoryService;

    private RoleService roleService;
    private Validator validator;

    private File importablePackageDirectory;

    private List<Importer> importers = Lists.newArrayList();

    @Autowired
    public ImporterServiceImpl(RepositoryService repositoryService, RoleService roleService, Validator validator) {
        this.repositoryService = repositoryService;
        this.roleService = roleService;
        this.validator = validator;
    }

    @PostConstruct
    public void initImporters() {
        Set<Class<? extends Importer>> importers = Scannit.getInstance().getSubTypesOf(Importer.class);
        logger.debug("Found importers: {}", importers);
        for (Class<? extends Importer> importer : importers) {
            try {
                if (importer.isInterface() || Modifier.isAbstract(importer.getModifiers()))
                    continue;
                if (importer.equals(XmlManifestDarImporter.class))
                    continue;
                logger.debug("Importer {} registered.", importer);
                this.importers.add(importer.newInstance());
            } catch (Exception e) {
                throw new IllegalStateException("Could not instantiate importer: " + importer, e);
            }
        }
        Collections.sort(this.importers, new Comparator<Object>() {
            @Override
            public int compare(Object o1, Object o2) {
                return o1.getClass().getSimpleName().compareTo(o2.getClass().getSimpleName());
            }
        });

        this.importers.add(new XmlManifestDarImporter(repositoryService));
        logger.debug("Importer {} registered.", XmlManifestDarImporter.class);

        logger.info("Importers configured in XL Deploy: {}", this.importers);
    }

    public void setImportablePackageDirectory(final File importablePackageDirectory) {
        this.importablePackageDirectory = importablePackageDirectory;
    }

    @Override
    public File getImportablePackageDirectory() {
        return importablePackageDirectory;
    }

    @Override
    public List<String> listPackages() {
        List<String> packagesFound = Lists.newArrayList();
        for (Importer importer : importers) {
            if (importer instanceof ListableImporter) {
                packagesFound.addAll(((ListableImporter) importer).list(importablePackageDirectory));
            }
        }

        Collections.sort(packagesFound);
        return packagesFound;
    }

    @Override
    public String importPackage(ImportSource source) {
        try {
            for (Importer importer : importers) {
                if (importer.canHandle(source)) {
                    return doImport(source, importer);
                }
            }
            throw new ImporterException("The selected file does not have the expected format for an importable package");
        } finally {
            source.cleanUp();
        }
    }

    private String doImport(ImportSource source, Importer importer) {
        ImportingContext ctx = new DefaultImportingContext();
        PackageInfo packageInfo = importer.preparePackage(source, ctx);
        try {
            boolean isUpgrade = isUpgrade(packageInfo);
            checkPermission(isUpgrade, packageInfo);
            checkImported(packageInfo);
            ImportedPackage importedPackage = importer.importEntities(packageInfo, ctx);
            scanPlaceholders(importedPackage, ctx);
            Set<ConfigurationItem> toCreate = newHashSet();
            createEntities(importedPackage, isUpgrade, toCreate);
            validate(toCreate);
            if (isUpgrade) {
                Distribution application = repositoryService.read(packageInfo.getApplicationId());
                importedPackage.getVersion().setDistribution(application);
            }
            RepositoryBaseCommand event = new CreateCisCommand(Lists.newArrayList(toCreate));
            publishCommand(event);
            return importedPackage.getVersion().getId();

        } finally {
            importer.cleanUp(packageInfo, ctx);
        }
    }

    private void publishCommand(final RepositoryBaseCommand event) {
        List<String> roles = listMyRoles();
        event.setSecurityContext(Permissions.getAuthenticatedUserName(), roles);
        EventBusHolder.publish(event, new ThrowingRuntimeExceptionHandlerStrategy());
    }

    private ArrayList<String> listMyRoles() {
        return newArrayList(Iterables.transform(roleService.getRolesFor(Permissions.getAuthentication()), new Function<Role, String>() {
            public String apply(final Role input) {
                return input.getName();
            }
        }));
    }

    private void scanPlaceholders(ImportedPackage importedPackage, ImportingContext ctx) {
        if (importedPackage.getVersion() instanceof DeploymentPackage) {
            File tempDir = Files.createTempDir();
            ctx.<List<TFile>>getAttribute("temporaryFiles").add(new TFile(tempDir));
            for (Deployable deployable : importedPackage.getDeployables()) {
                if (deployable instanceof SourceArtifact) {
                    SourceArtifact sourceArtifact = (SourceArtifact) deployable;
                    ArtifactEnricher artifactEnricher = new ArtifactEnricher(sourceArtifact, new WorkDir((LocalFile) LocalFile.valueOf(tempDir)));
                    try {
                        artifactEnricher.enrich();
                    } catch (Exception e) {
                        throw new ImporterException(e);
                    }
                }
            }
        }
    }


    private void checkImported(PackageInfo packageInfo) {
        if (repositoryService.exists(IdGenerator.generateId(packageInfo.getApplicationId(), packageInfo.getApplicationVersion()))) {
            throw new ImporterException("Already imported version %s of application %s", packageInfo.getApplicationVersion(), packageInfo.getApplicationName());
        }
        String dirs = subStringBeforeLast(packageInfo.getApplicationId(), "/");
        if (!repositoryService.exists(dirs) ) {
            throw new ImporterException("The directory structure [%s] specified for the import of application [%s] does not exist.", dirs, subStringAfterLast(packageInfo.getApplicationId(), "/"));
        }
    }

    private void createEntities(ImportedPackage importedPackage, boolean isUpgrade, final Set<ConfigurationItem> toCreateCollector) {
        if (!isUpgrade) {
            toCreateCollector.add(importedPackage.getApplication());
        }
        DistributionVersion version = importedPackage.getVersion();
        toCreateCollector.add(version);
        version.getType().getDescriptor().getPropertyDescriptors().stream().forEach(pd -> {
            boolean isSetOfCI = pd.getKind() == PropertyKind.SET_OF_CI;
            boolean isListOfCI = pd.getKind() == PropertyKind.LIST_OF_CI;
            if (isSetOfCI || isListOfCI) {
                Collection<ConfigurationItem> cis = (Collection<ConfigurationItem>) pd.get(version);
                toCreateCollector.addAll(newHashSet(cis));
                cis.forEach(ci -> createNestedConfigurationItems(ci, toCreateCollector));
            }
        });
    }

    private void createNestedConfigurationItems(ConfigurationItem ci, final Set<ConfigurationItem> toCreateCollector) {
        Collection<PropertyDescriptor> propertyDescriptors = ci.getType().getDescriptor().getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            boolean isSetOfCI = propertyDescriptor.getKind() == PropertyKind.SET_OF_CI;
            boolean isListOfCI = propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI;
            if (isSetOfCI || isListOfCI &&
                    propertyDescriptor.getReferencedType().getDescriptor().isAssignableTo(EmbeddedDeployable.class)) {

                Collection<ConfigurationItem> embeddeds = (Collection<ConfigurationItem>) propertyDescriptor.get(ci);
                for (ConfigurationItem embedded : embeddeds) {
                    if (!toCreateCollector.contains(embedded)) {
                        toCreateCollector.add(embedded);
                        createNestedConfigurationItems(embedded, toCreateCollector);
                    }
                }
            }
        }
    }


    private void validate(Set<ConfigurationItem> toCreate) {
        List<ValidationMessage> msgs = newArrayList();

        for (ConfigurationItem toCreateEntity : toCreate) {
            msgs.addAll(validator.validate(toCreateEntity, newArrayList(toCreate)));
        }

        if (!msgs.isEmpty()) {
            throw new ImporterException("Import failed with the following validation errors %s", msgs.toString());
        }
    }

    private boolean isUpgrade(PackageInfo packageInfo) {
        String appId = null;
        String appName = subStringAfterLast(packageInfo.getApplicationName(), "/");
        boolean locationSpecified = (packageInfo.getApplicationName().contains("/"));

        List<ConfigurationItemData> applications = repositoryService.list(new SearchParameters().setType(Type.valueOf(Distribution.class)).setName(appName).setAncestor(packageInfo.getApplicationRoot()));
        for (ConfigurationItemData application : applications) {
            checkState(appId == null, "Found more than 1 [%s] with the same name: [%s] and [%s]",application.getType(), appId, application);
            appId = application.getId();
        }

        if (appId != null) {
            if (locationSpecified && !appId.equals(packageInfo.getApplicationId())) {
                throw new ImporterException("The manifest contains the path [%s] to import the application into, but the application exists at the path [%s]", packageInfo.getApplicationId(), appId);
            }
            packageInfo.setDirectories(appId.substring(packageInfo.getApplicationRoot().length(), appId.length() - packageInfo.getApplicationName().length()));
            return true;
        }

        return false;
    }

    private String subStringAfterLast(String s, String sep) {
        if (s.contains(sep)) {
            return s.substring(s.lastIndexOf(sep) + sep.length());
        }
        return s;
    }

    private String subStringBeforeLast(String s, String sep) {
        if (s.contains(sep)) {
            return s.substring(0, s.lastIndexOf(sep));
        }
        return s;
    }

    private void checkPermission(boolean isUpgrade, PackageInfo packageInfo) {
        if (isUpgrade) {
            checkPermission(IMPORT_UPGRADE, packageInfo.getApplicationId());
        } else {
            checkPermission(IMPORT_INITIAL, packageInfo.getApplicationId());
        }
    }

    void checkPermission(Permission permission, String onConfigurationItems) {
        if (!permission.getPermissionHandler().hasPermission(onConfigurationItems)) {
            throw PermissionDeniedException.forPermission(permission, onConfigurationItems);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ImporterServiceImpl.class);

}
