package com.xebialabs.deployit.service.deployment.setter;

import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.DictionaryValueException;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Set;

@Component
public class FallbackPropertySetter extends AbstractPropertySetter {
    private static final Logger logger = LoggerFactory.getLogger(FallbackPropertySetter.class);

    @Autowired
    public FallbackPropertySetter(RepositoryService repository) {
        super(repository);
    }

    private void checkWhetherValueIsSecureAndPropertyIsPassword(Object value, PropertyDescriptor deployedDesc, Object deployablePropertyValue) throws DictionaryValueException {
        if (value instanceof EncryptedStringValue && !deployedDesc.isPassword()) {
            logger.warn("The deployable value [{}] resolved to an encrypted value, but property [{}] is not a password field.", deployablePropertyValue, deployedDesc.getFqn());
            throw new DictionaryValueException("The deployable value [%s] resolved to an encrypted value, but property [%s] is not a password field.", deployablePropertyValue, deployedDesc.getFqn());
        }
    }

    @Override
    public void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                            ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor,
                            Set<String> missingPlaceholdersAggregator) {
        String name = targetPropertyDescriptor.getName();
        Object previousTargetValue = getValueIfExists(previousTarget, name);
        Object previousSourceValue = getValueIfExists(previousSource, name);
        Object currentSourceValue = getValueIfExists(currentSource, name);

        PropertyDescriptor sourcePropDesc = currentSource.getType().getDescriptor().getPropertyDescriptor(name);

        Object valueToSet = previousTargetValue;
        PropertyDescriptor valueToSetPropDesc = targetPropertyDescriptor;

        if (previousSourceValue != null && MustachePlaceholderScanner.hasPlaceholders(previousSourceValue, sourcePropDesc)) {
            // If the old deployable value had placeholder, always use the newly set one.
            valueToSet = currentSourceValue;
            valueToSetPropDesc = sourcePropDesc;
        } else if (previousSourceValue == null && currentSourceValue != null) {
            // If the old deployable value was not set, and now it is, use the newly set one.
            valueToSet = currentSourceValue;
            valueToSetPropDesc = sourcePropDesc;
        } else if (previousSourceValue != null) {
            // Check whether the old deployable value resolves to the same as the currently deployed value
            PropertyDescriptor previousSourcePropDesc = previousSource.getType().getDescriptor().getPropertyDescriptor(name);
            Object resolvedPreviousSourceValue = convertIfNeeded(previousSourceValue, previousSourcePropDesc, targetPropertyDescriptor);
            if (previousTargetValue != null && resolvedPreviousSourceValue.toString().equals(previousTargetValue.toString())) {
                // If the resolved old deployable value equals the old deployed value, use the newly set one.
                valueToSet = currentSourceValue;
                valueToSetPropDesc = sourcePropDesc;
            }
        }

        if (isNullOrEmpty(valueToSet, valueToSetPropDesc)) {
            setDeployedFromDictionary(currentTarget, dictionary, targetPropertyDescriptor, sourcePropDesc);
            return;
        }

        try {
            Object deployedPropertyValue = dictionary.resolve(valueToSet, valueToSetPropDesc);
            checkWhetherValueIsSecureAndPropertyIsPassword(deployedPropertyValue, targetPropertyDescriptor, valueToSet);
            setSilently(currentTarget, targetPropertyDescriptor, convertIfNeeded(deployedPropertyValue, valueToSetPropDesc, targetPropertyDescriptor));
        } catch (DictionaryValueException e) {
            setSilently(currentTarget, targetPropertyDescriptor, previousTargetValue);
            logger.info("Could not resolve dictionary keys for property " + sourcePropDesc + " on " + currentTarget + ".");
            logger.trace("Exception was", e);
        }
    }
}
