package com.xebialabs.deployit.service.deployment

import java.util

import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.engine.spi.execution.Transitions._
import com.xebialabs.deployit.engine.spi.execution.{NonRemotableExecutionStateListener, StepExecutionStateEvent, TaskExecutionStateEvent}
import com.xebialabs.deployit.engine.tasker.TaskStep
import com.xebialabs.deployit.plugin.api.flow.Step
import org.slf4j.LoggerFactory

import scala.beans.BeanProperty
import scala.collection.convert.wrapAll._

class CheckPointManagerListener(@BeanProperty val fullSpec: MultiDeltaSpecification, checkpoints: util.List[StepPlan.Checkpoint]) extends NonRemotableExecutionStateListener {
  import CheckpointManager._

  @transient private val logger = LoggerFactory.getLogger(getClass)

  @BeanProperty val checkpointManager = new CheckpointManager(fullSpec)

  override def taskStateChanged(event: TaskExecutionStateEvent) {
    if (checkTransition(event, STOPPED, CANCELLED) ||
      checkTransition(event, FAILED, CANCELLED) ||
      checkTransition(event, ABORTED, CANCELLED) ||
      checkTransition(event, CANCELLING, CANCELLED)) {
      logger.info("Partial commit triggered")
      checkpointManager.doPartialCommit()
    }
  }

  override def stepStateChanged(event: StepExecutionStateEvent) {
    logger.debug(s"Got event [$event]")
    val step: TaskStep = event.step().asInstanceOf[TaskStep]
    // SKIPPED or DONE steps are checkpointed.
    if (!step.getState.isFinal) {
      return
    }

    step.getMetadata.filterKeys(_.startsWith("checkpointId")).toList.sortBy(_._1).foreach { case (k, cpId) =>
      markCheckpoints((cpId, step.getImplementation))
    }
  }

  private[this] def markCheckpoints(tuple: (String, Step)) = tuple match {
    case (_, repo: RepositoryUpdateStep) =>
      // if the step is a RepositoryUpdateStep there is no checkpoint to set, so we mark the DeltaSpecification as affected.
      checkpointManager.markDeployedApplication(repo.spec.getAppliedDistribution.getId)
    case (null, _) => // No checkpoint is needed.
    case (checkpointId, _) =>
      val toMark = checkpoints.view.filter(_.getId == checkpointId)
      toMark.foreach { cp =>
        val delta = cp.getDelta
        val deployedId = delta.deployed.getId

        checkpointManager.markCheckpoint(deployedId, cp.getOperation, cp.getIntermediateCheckpointName)
      }
  }
}

