package com.xebialabs.deployit.service.deployment

import java.util

import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder._
import com.xebialabs.deployit.deployment.planner.{DefaultDelta, DeltaSpecificationBuilder, MultiDeltaSpecification}
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Operation}
import com.xebialabs.deployit.plugin.api.reflect.Descriptor
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution
import com.xebialabs.deployit.plugin.api.udm.{Container, Deployable, Deployed}
import com.xebialabs.deployit.repository.{ChangeSet, RepositoryServiceHolder}
import org.slf4j.LoggerFactory

import scala.collection.mutable

class CheckpointManager(fullSpec: MultiDeltaSpecification) extends Serializable {

  import CheckpointManager._

  import collection.convert.wrapAll._

  @transient private val logger = LoggerFactory.getLogger(getClass)

  private[this] val checkpointDeltas: Map[String, CheckpointDelta] = fullSpec.getDeltaSpecifications.flatMap { ds =>
    ds.getDeltas.map(d => (d.deployed.getId, new CheckpointDelta(d, ds)))
  }.toMap

  private[this] val checkpointDeltaSpecifications = fullSpec.getDeltaSpecifications
    .map(ds => ds.deployedApplication.getId -> new CheckpointDeltaSpecification(ds, affected = false)).toMap


  def markDeployedApplication(deployedApplicationId: String) = checkpointDeltaSpecifications(deployedApplicationId).affected = true

  def markCheckpoint(id: String, operation: Operation, intermediateCheckpointName: String): Unit = {
    val checkpointDelta: CheckpointDelta = checkpointDeltas(id)
    checkpointDelta.markCheckpoint(operation, intermediateCheckpointName)
    markDeployedApplication(checkpointDelta.ds.deployedApplication.getId)
  }

  def prepareRollback(): MultiDeltaSpecification = {
    def determineDeltaSpecificationOperation(builder: DeltaSpecificationBuilder,
                                             operation: Operation,
                                             deployedApplication: AppliedDistribution,
                                             previouslyDeployedApplication: AppliedDistribution,
                                             destroyedDeployeds: Set[Deployed[_ <: Deployable, _ <: Container]]) {
      operation match {
        case CREATE =>
          builder.undeploy(deployedApplication)
        case DESTROY if previouslyDeployedApplication.getDeployeds.toSet.diff(destroyedDeployeds).isEmpty =>
          logger.info("The undeployment was rolled back after completion, fully redeploying the previous deployment.")
          builder.initial(previouslyDeployedApplication)
        case DESTROY =>
          builder.upgrade(deployedApplication, previouslyDeployedApplication)
        case MODIFY if deployedApplication.getDeployeds.toSet.diff(destroyedDeployeds).isEmpty =>
          logger.info("The update deployment was rolled back when there were no deployeds left, fully redeploying the previous deployment.")
          builder.initial(previouslyDeployedApplication)
        case MODIFY =>
          builder.upgrade(deployedApplication, previouslyDeployedApplication)
        case NOOP =>
          throw new IllegalStateException("DeltaSpecification can never have a NOOP operation.")
      }
    }

    val specs = checkedSpecsToDeltas(includeAll).view.reverseMap { case (spec, checkpoints) =>
      val builder: DeltaSpecificationBuilder = newRollbackSpecification
      val destroyedDeployeds = mutable.Set[Deployed[_ <: Deployable, _ <: Container]]()
      checkpoints.foreach { cp =>
        cp.getOperation match {
          case Operation.DESTROY if cp.fullyCheckpointed =>
            destroyedDeployeds.add(cp.deployed)
            cp.rollback().foreach(builder.`with`)
          case _ =>
            cp.rollback().foreach(builder.`with`)
        }
      }

      determineDeltaSpecificationOperation(builder, spec.getOperation, spec.getAppliedDistribution, spec.getPreviousAppliedDistribution, destroyedDeployeds.toSet)
      builder.build

    }.toList

    MultiDeltaSpecification.forRollback(specs)
  }

  private[this] def isMarkedDelta(delta: CheckpointDelta) = delta.fullyCheckpointed || delta.getIntermediateCheckpoints.nonEmpty

  private[this] def includeAll(delta: CheckpointDelta) = true

  private[this] def checkedSpecsToDeltas(filter:CheckpointDelta => Boolean): List[(DeltaSpecification, List[CheckpointDelta])] = {
    val affectedSpecs = checkpointDeltaSpecifications.collect { case (id, cds) if cds.affected => cds.ds }
    val mapped = affectedSpecs.map(spec => spec -> checkpointDeltas.values.collect { case cd if filter(cd) && cd.ds == spec => cd }.toList).toList
    //Return at least main application when no checkpoints are found
    if (mapped.nonEmpty) mapped else List(fullSpec.getMainDeltaSpecification -> Nil)
  }

  def doPartialCommit(): Unit = {
    val changeSet = new ChangeSet
    checkedSpecsToDeltas(isMarkedDelta).foreach { case (spec, deltas) =>
      val partialDeployedApplication = getPartialDeployedApplication(spec)
      deltas.foreach { cp =>
        cp.getOperation match {
          case CREATE =>
            partialDeployedApplication.addDeployed(cp.getDeployed)
            changeSet.createOrUpdate(cp.getDeployed)
          case DESTROY =>
            partialDeployedApplication.getDeployeds.remove(cp.getPrevious)
            changeSet.delete(cp.getPrevious)
          case MODIFY | NOOP =>
            partialDeployedApplication.getDeployeds.remove(cp.getPrevious)
            partialDeployedApplication.addDeployed(cp.getDeployed)
            changeSet.createOrUpdate(cp.getDeployed)
        }
      }

      if (partialDeployedApplication.getDeployeds.isEmpty) {
        changeSet.delete(partialDeployedApplication)
      } else {
        changeSet.createOrUpdate(partialDeployedApplication)
      }
    }

    logger.info(s"Storing the partial commit for ${fullSpec.getMainDeployedApplication.getId}")
    RepositoryServiceHolder.getRepositoryService.execute(changeSet)
    logger.info(s"Partial commit for ${fullSpec.getMainDeployedApplication.getId} succeeded")
  }

  private[this] def duplicateDeployedApplication(deployedApplication: AppliedDistribution) = {
    val descriptor: Descriptor = deployedApplication.getType.getDescriptor
    val nda: AppliedDistribution = descriptor.newInstance(deployedApplication.getId)
    descriptor.getPropertyDescriptors.filterNot(_.getName == "deployeds").foreach { pd =>
      pd.set(nda, pd.get(deployedApplication))
    }
    nda.setDeployeds(new util.HashSet(deployedApplication.getDeployeds))
    nda
  }

  private[this] def getPartialDeployedApplication(spec: DeltaSpecification) = spec.getOperation match {
    case CREATE =>
      val nda = duplicateDeployedApplication(spec.getAppliedDistribution)
      nda.getDeployeds.clear()
      nda
    case _ => duplicateDeployedApplication(spec.getPreviousAppliedDistribution)
  }
}

private class CheckpointDeltaSpecification(val ds: DeltaSpecification, var affected: Boolean) extends Serializable

private class CheckpointDelta(d: Delta, val ds: DeltaSpecification) extends Delta {

  import CheckpointManager._

  import collection.convert.wrapAll._

  @transient private lazy val logger = LoggerFactory.getLogger(getClass)

  var fullyCheckpointed: Boolean = false
  private val intermediateCheckpoints = mutable.ListBuffer[String]()
  private var overrideOperation: Option[Operation] = None

  override def getDeployed: Deployed[_ <: Deployable, _ <: Container] = d.getDeployed.asInstanceOf[Deployed[Deployable, Container]]

  override def getOperation: Operation = overrideOperation.getOrElse(d.getOperation)

  override def getPrevious: Deployed[_ <: Deployable, _ <: Container] = d.getPrevious.asInstanceOf[Deployed[Deployable, Container]]

  /* Do not go to super, the intermediateCheckpoints is immutable */
  override def getIntermediateCheckpoints: util.List[String] = intermediateCheckpoints

  def markCheckpoint(operation: Operation, intermediateCheckpointName: String): Unit = {
    overrideOperation = calculateOverrideOperation(operation)

    Option(intermediateCheckpointName).filterNot(_.isEmpty) match {
      case Some(x) if !fullyCheckpointed =>
        logger.info(s"Marked intermediate checkpoint $x for ${this.deployed.getId}")
        intermediateCheckpoints.add(x)
      case Some(x) =>
        logger.info(s"Cannot mark intermediate checkpoint $x for ${this.deployed.getId} as it is already fully checkpointed")
      case None =>
        logger.info(s"Marked checkpoint for ${this.deployed.getId}")
        intermediateCheckpoints.clear()
        fullyCheckpointed = true
    }
  }

  def calculateOverrideOperation(operation: Operation): Option[Operation] = {
    (d.getOperation, operation) match {
      case (MODIFY, DESTROY) => Some(DESTROY)
      case (MODIFY, CREATE) if overrideOperation.contains(DESTROY) => None
      case (x, y) if y != null && x != y =>
        logger.error(s"Cannot mark checkpoint with operation $y for delta ${d.deployed.getId} with operation $x")
        // Plugin provided a wrong override operation, ignore it.
        None
      case _ =>
        // Delta operation and checkpoint operation match, or checkpoint had a null operation
        None
    }
  }

  def rollback(): Option[Delta] = {
    val hasHitCheckpoint: Boolean = fullyCheckpointed || intermediateCheckpoints.nonEmpty

    import Deltas._
    getOperation match {
      case Operation.CREATE if !hasHitCheckpoint =>
        None
      // A create that has hit no checkpoint has no rollback variant
      case _ if !hasHitCheckpoint =>
        // Any other operation which has not hit a checkpoint is a NOOP in a rollback
        Some(noOp(getPrevious))
      case Operation.CREATE =>
        // A create which hit a checkpoint is a DESTROY (with possible intermediate checkpoints)
        Some(destroy(getDeployed, intermediateCheckpoints.toList))
      case Operation.DESTROY if hasHitCheckpoint =>
        // A Destroy which is hit a checkpoint is a CREATE with checkpoints
        Some(create(getPrevious, intermediateCheckpoints.toList))
      case Operation.MODIFY if hasHitCheckpoint =>
        Some(modify(getDeployed, getPrevious, intermediateCheckpoints.toList))
      case _ =>
        Some(noOp(getPrevious))
    }
  }
}


object CheckpointManager {

  implicit class DeltaUtils(val delta: Delta) extends AnyVal {
    def deployed = if (delta.getDeployed != null) delta.getDeployed else delta.getPrevious
  }

  implicit class DeltaSpecUtils(val deltaSpec: DeltaSpecification) extends AnyVal {
    def deployedApplication = if (deltaSpec.getAppliedDistribution != null) {
      deltaSpec.getAppliedDistribution
    } else {
      deltaSpec.getPreviousAppliedDistribution
    }
  }

}

object Deltas {

  import collection.convert.wrapAll._

  def destroy(d: Deployed[_ <: Deployable, _ <: Container], intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(DESTROY, d, null, intermediateCheckpoints)
  }

  def create(d: Deployed[_ <: Deployable, _ <: Container], intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(CREATE, null, d, intermediateCheckpoints)
  }

  def modify(previous: Deployed[_ <: Deployable, _ <: Container],
             wanted: Deployed[_ <: Deployable, _ <: Container],
             intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(MODIFY, previous, wanted, intermediateCheckpoints)
  }

  def noOp(d: Deployed[_ <: Deployable, _ <: Container], intermediateCheckpoints: List[String] = Nil): Delta = {
    new DefaultDelta(NOOP, d, d, intermediateCheckpoints)
  }
}