package com.xebialabs.deployit.service

import com.xebialabs.deployit.plugin.api.semver.VersionRange
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage

import scala.util.Try
import scalax.collection.GraphPredef.Param
import scalax.collection.edge.LDiEdge
import scalax.collection.{mutable => xmutable}
import com.github.zafarkhaja.semver.{Version => OsgiVersion}

package object dependency {
  private[dependency] type OutboundDependencies = xmutable.Graph[Application, LDiEdge]

  private[dependency] def graph(elems: Param[Application, LDiEdge]*): OutboundDependencies = {
    xmutable.Graph[Application, LDiEdge](elems: _*)
  }

  implicit class RangeUtils(val osgiVersion: OsgiVersion) extends AnyVal {
    def in(versionRange: VersionRange) = if (versionRange.includes(osgiVersion)) Option(osgiVersion) else None
  }

  implicit class VersionUtils(val v1: Option[OsgiVersion]) extends AnyVal {
    def max(v2: Option[OsgiVersion]) = Try(Seq(v1, v2).flatten.max).toOption
  }

  implicit class UdmVersionUtils(val v1: DeploymentPackage) extends AnyVal {

    def toOsgi = parseVersion(v1.getName)
  }

  implicit class OutboundDependenciesUtils(val graph: OutboundDependencies) extends AnyVal {
    def allRequiredVersionsInTopologicalOrder = graph.topologicalSort.flatMap(app => graph.get(app).incoming.headOption.flatMap(_.label.asInstanceOf[Label].udmVersion))

    def version(app: Application) = graph.get(app).incoming.head.label.asInstanceOf[Label].udmVersion.get
  }

  implicit val osgiOrdering: Ordering[OsgiVersion] = Ordering.fromLessThan((v1, v2) => v1.compareTo(v2) < 0)

  def parseVersion(version: String) = {
    val parts = version.split("\\.")
    if (parts.length == 1) {
      OsgiVersion.forIntegers(parts(0).toInt)
    } else if (parts.length == 2) {
      OsgiVersion.forIntegers(parts(0).toInt,parts(1).toInt)
    } else {
      OsgiVersion.valueOf(version)
    }
  }

}
