package com.xebialabs.deployit.service.deployment;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.EncryptedStringValue;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployable;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployedContainer;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary;
import com.xebialabs.deployit.service.replacement.DictionaryValueException;
import com.xebialabs.deployit.service.replacement.MustachePlaceholderScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.plugin.api.validation.ValidationMessage.warn;

@Component
public class DeployedPropertySetter {

    private RepositoryService repository;

    private final TypeCalculator calculator;

    private final ServerConfiguration serverConfiguration;

    @Autowired
    public DeployedPropertySetter(RepositoryService repository, TypeCalculator typeCalculator, ServerConfiguration serverConfiguration) {
        this.repository = repository;
        this.calculator = typeCalculator;
        this.serverConfiguration = serverConfiguration;
    }

    public void setProperties(EmbeddedDeployedContainer<?, ?> deployed, EmbeddedDeployedContainer<?, ?> previousDeployed, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        checkArgument(previousDeployed == null || deployed.getType().equals(previousDeployed.getType()), "New and existing target types should be equal when upgrading (%s)", deployed.getType());
        for (PropertyDescriptor deployedPropDesc : deployed.getType().getDescriptor().getPropertyDescriptors()) {

            if (deployedPropDesc.isDeployedSpecific()) {
                continue;
            }

            if (isEmbeddedProperty(deployedPropDesc)) {
                generateEmbeddeds(deployed, deployedPropDesc, dictionary, missingPlaceholdersAggregator);
                continue;
            }

            final ConfigurationItem currentDeployable = deployed.getDeployable();
            ConfigurationItem previousDeployable = previousDeployed == null ? null : previousDeployed.getDeployable();
            setProperty(deployed, previousDeployed, currentDeployable, previousDeployable, dictionary, deployedPropDesc, missingPlaceholdersAggregator);
        }
    }

    public void setProperties(AppliedDistribution currentDeployedApp, AppliedDistribution existingDeployedApp, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        final ConfigurationItem previousDeployable = existingDeployedApp == null ? null : existingDeployedApp.getVersion();
        final ConfigurationItem currentDeployable = currentDeployedApp.getVersion();

        for (PropertyDescriptor deployedPropDesc : currentDeployedApp.getType().getDescriptor().getPropertyDescriptors()) {

            if (deployedPropDesc.isDeployedSpecific()) {
                continue;
            }
            setProperty(currentDeployedApp, existingDeployedApp, currentDeployable, previousDeployable, dictionary, deployedPropDesc, missingPlaceholdersAggregator);
        }
        for (PropertyDescriptor pd : currentDeployable.getType().getDescriptor().getPropertyDescriptors()) {
            if (currentDeployedApp.hasProperty(pd.getName())) {
                PropertyDescriptor deployedAppPropertyDescriptor = currentDeployedApp.getType().getDescriptor().getPropertyDescriptor(pd.getName());
                setProperty(currentDeployedApp, existingDeployedApp, currentDeployable, previousDeployable, dictionary, deployedAppPropertyDescriptor, missingPlaceholdersAggregator);
            }
        }

    }

    private void setProperty(ConfigurationItem currentTarget, ConfigurationItem previousTarget, ConfigurationItem currentSource,
                             ConfigurationItem previousSource, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor, Set<String> missingPlaceholdersAggregator) {

        logger.debug("Going to set property {} of currentTarget {}", targetPropertyDescriptor.getFqn(), currentTarget);

        String propertyName = targetPropertyDescriptor.getName();

        if (!serverConfiguration.isServerMappingOverrideDeployedFieldsOnUpdate() && !isPropertyValueEmptyOrNull(previousTarget, propertyName)) {
            logger.debug("{} on targetNode contained value and backward comparability flag is turned on. Leaving it as-is.", propertyName);
            copyFromPreviousDeployed(currentTarget, previousTarget, propertyName);
        } else if (currentSource.hasProperty(propertyName) && !isPropertyValueEmptyOrNull(currentSource, propertyName)) {
            logger.debug("Resolving '{}' value from currentSource and setting the value on the current currentTarget", propertyName);
            resolveFromSource(currentSource, currentTarget, targetPropertyDescriptor, dictionary, missingPlaceholdersAggregator);
        } else if (dictionary.containsKey(targetPropertyDescriptor.getFqn())) {
            logger.debug("Resolving {} from dictionary as dictionary contains a type default for the property of the currentTarget", propertyName);
            resolveFromDictionary(currentTarget, currentSource, dictionary, targetPropertyDescriptor, propertyName);
        } else if (!isPropertyValueEmptyOrNull(currentTarget, propertyName)) {
            logger.debug("Using type system default as currentTarget '{}' has non-empty default value", propertyName);
        } else {
            if (!currentSource.hasProperty(propertyName) || isPropertyOnPreviousAndCurrentDeployableEmptyOrNull(currentSource, previousSource, propertyName)) {
                logger.debug("copy previousTarget '{}' value to current currentTarget value", propertyName);
                copyFromPreviousDeployed(currentTarget, previousTarget, propertyName);
            } else {
                logger.debug("leave currentTarget '{}' empty", propertyName);
            }
        }

    }

    private boolean isPropertyOnPreviousAndCurrentDeployableEmptyOrNull(ConfigurationItem currentDeployable, ConfigurationItem previousDeployable, String propertyName) {
        return isPropertyValueEmptyOrNull(currentDeployable, propertyName) && isPropertyValueEmptyOrNull(previousDeployable, propertyName);
    }

    private void copyFromPreviousDeployed(ConfigurationItem currentDeployed, ConfigurationItem previousDeployed, String propertyName) {
        Object propertyValue = getValueIfExists(previousDeployed, propertyName);
        currentDeployed.setProperty(propertyName, propertyValue);
    }

    private void resolveFromDictionary(ConfigurationItem currentDeployed, ConfigurationItem currentDeployable, ConsolidatedDictionary dictionary, PropertyDescriptor deployedPropDesc, String propertyName) {
        Object value = dictionary.get(deployedPropDesc.getFqn());
        PropertyDescriptor currentDeployablePropDesc = currentDeployable.getType().getDescriptor().getPropertyDescriptor(propertyName);
        value = convertIfNeeded(value, currentDeployablePropDesc, deployedPropDesc);
        deployedPropDesc.set(currentDeployed, value);
    }

    private boolean isPropertyValueEmptyOrNull(ConfigurationItem ci, String propertyName) {
        if (ci == null || !ci.hasProperty(propertyName)) {
            return true;
        }
        Object value = ci.getProperty(propertyName);
        PropertyDescriptor propertyDescriptor = ci.getType().getDescriptor().getPropertyDescriptor(propertyName);
        return isNullOrEmpty(value, propertyDescriptor);
    }

    private void resolveFromSource(ConfigurationItem source, ConfigurationItem target, PropertyDescriptor targetPropertyDescriptor, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        String propertyName = targetPropertyDescriptor.getName();
        PropertyDescriptor sourcePropertyDescriptor = source.getType().getDescriptor().getPropertyDescriptor(propertyName);

        // Resolve from source
        logger.debug("Resolving property {} from source {} with type {}", targetPropertyDescriptor.getFqn(), source.getId(), source.getType());
        Object deployableValue = source.getProperty(propertyName);
        // If source has no value, try dictionary propertyname fallback
        if (isNullOrEmpty(deployableValue, sourcePropertyDescriptor)) {
            logger.debug("Checking whether {} occurs in dictionary", targetPropertyDescriptor.getFqn());
            setDeployedFromDictionary(target, dictionary, targetPropertyDescriptor, sourcePropertyDescriptor);
            return;
        }
        // resolve source value through dictionary
        resolveSourceValueFromDictionary(target, source, dictionary, targetPropertyDescriptor, missingPlaceholdersAggregator);
    }

    private void resolveSourceValueFromDictionary(ConfigurationItem target, ConfigurationItem source, ConsolidatedDictionary dictionary, PropertyDescriptor targetPropertyDescriptor, Set<String> missingPlaceholdersAggregator) {
        String propertyName = targetPropertyDescriptor.getName();
        Object sourceValue = source.getProperty(propertyName);
        try {
            PropertyDescriptor sourcePropertyDescriptor = source.getType().getDescriptor().getPropertyDescriptor(propertyName);
            Object targetValue = dictionary.resolve(sourceValue, sourcePropertyDescriptor);
            if (targetValue instanceof EncryptedStringValue && !targetPropertyDescriptor.isPassword()) {
                logger.warn("The source value [{}] resolved to an encrypted value, but property [{}] is not a password field.", sourceValue, targetPropertyDescriptor.getFqn());
                ValidationMessage validationMessage = warn(target.getId(), propertyName, "The deployable value resolved to an encrypted value, but this field is not a password");
                target.get$validationMessages().add(validationMessage);
                return;
            }
            setSilently(target, targetPropertyDescriptor, convertIfNeeded(targetValue, sourcePropertyDescriptor, targetPropertyDescriptor));
        } catch (DictionaryValueException e) {
            logger.error("Could not resolve dictionary keys for property " + targetPropertyDescriptor + " on " + target + ".");
            missingPlaceholdersAggregator.addAll(e.getMissingValues());
            target.get$validationMessages().add(warn(target.getId(), propertyName, "Could not resolve the deployable value \"" + sourceValue + "\" using the dictionary."));
        }
    }

    private void setSilently(ConfigurationItem ci, PropertyDescriptor pd, Object value) {
        try {
            pd.set(ci, value);
        } catch (RuntimeException exc) {
            logger.error("Could not convert (resolved) value to correct type for property " + pd + " of " + ci + ". Error message was: " + exc.getMessage());
            ci.get$validationMessages().add(warn(ci.getId(), pd.getName(), "Could not correctly convert resolved value, property empty"));
        }
    }

    private Object getValueIfExists(ConfigurationItem configurationItem, String name) {
        if (configurationItem == null || !configurationItem.hasProperty(name)) {
            return null;
        }

        return configurationItem.getProperty(name);
    }

    private boolean isNullOrEmpty(final Object value, final PropertyDescriptor pd) {
        if (value == null) return true;
        switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
            case ENUM:
            case DATE:
            case CI:
                return Strings.isNullOrEmpty(value.toString());
            case SET_OF_STRING:
            case SET_OF_CI:
            case LIST_OF_STRING:
            case LIST_OF_CI:
                return ((Collection<?>) value).isEmpty();
            case MAP_STRING_STRING:
                return ((Map<?, ?>) value).isEmpty();
            default:
                throw new IllegalStateException("Unsupported property kind: " + pd.getKind() + " for property: " + pd.getFqn());
        }
    }

    private void setDeployedFromDictionary(ConfigurationItem deployed, ConsolidatedDictionary dictionary, PropertyDescriptor deployedPropDesc, PropertyDescriptor deployablePropDesc) {
        if (dictionary.containsKey(deployedPropDesc.getFqn())) {
            Object value = dictionary.get(deployedPropDesc.getFqn());
            value = convertIfNeeded(value, deployablePropDesc, deployedPropDesc);
            deployedPropDesc.set(deployed, value);
        }
    }

    private Object convertIfNeeded(Object valueToSet, PropertyDescriptor deployablePropDesc, PropertyDescriptor deployedPropDesc) {
        if (valueToSet == null) return null;
        if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.CI) {
            return repository.read(valueToSet.toString());
        } else if (optionalKind(deployablePropDesc, PropertyKind.SET_OF_STRING) && deployedPropDesc.getKind() == PropertyKind.SET_OF_CI) {
            return Sets.newHashSet(getCollectionOfCis(valueToSet));
        } else if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.SET_OF_CI) {
            return Sets.newHashSet(getCollectionOfCis(splitValue(valueToSet)));
        } else if (optionalKind(deployablePropDesc, PropertyKind.LIST_OF_STRING) && deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI) {
            return Lists.newArrayList(getCollectionOfCis(valueToSet));
        } else if (optionalKind(deployablePropDesc, PropertyKind.STRING) && deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI) {
            return Lists.newArrayList(getCollectionOfCis(splitValue(valueToSet)));
        } else if (deployedPropDesc.getKind() == PropertyKind.STRING) {
            if (valueToSet instanceof EncryptedStringValue && !deployedPropDesc.isPassword()) {
                logger.warn("Going to write a Encrypted value as plaintext in \"{}\", as it is not a password field.", deployedPropDesc);
            }
            return valueToSet.toString();
        } else if (valueToSet instanceof StringValue) {
            return valueToSet.toString();
        }
        return valueToSet;
    }

    private Collection<String> splitValue(Object valueToSet) {
        checkArgument((valueToSet instanceof String) || (valueToSet instanceof StringValue), "Should be String or StringValue to split.");
        Splitter splitter = Splitter.on(",").omitEmptyStrings();
        return newArrayList(splitter.split(valueToSet.toString()));
    }

    private boolean optionalKind(PropertyDescriptor deployablePropDesc, PropertyKind kind) {
        return deployablePropDesc == null || deployablePropDesc.getKind() == kind;
    }

    @SuppressWarnings("unchecked")
    private Collection<ConfigurationItem> getCollectionOfCis(Object valueToSet) {
        Collection<ConfigurationItem> cis = newArrayList();
        for (String s : (Collection<String>) valueToSet) {
            cis.add(repository.read(s));
        }
        return cis;
    }

    private void generateEmbeddeds(EmbeddedDeployedContainer<?, ?> deployed, final PropertyDescriptor deployedPropDesc, ConsolidatedDictionary dictionary, Set<String> missingPlaceholdersAggregator) {
        final ConfigurationItem deployable = deployed.getDeployable();
        String name = deployedPropDesc.getName();
        Collection<EmbeddedDeployed<?, ?>> embeddedDeployeds = deployedPropDesc.getKind() == PropertyKind.LIST_OF_CI ? Lists.<EmbeddedDeployed<?, ?>>newArrayList() : Sets.<EmbeddedDeployed<?, ?>>newHashSet();
        PropertyDescriptor deployableProperty = deployable.getType().getDescriptor().getPropertyDescriptor(name);
        if (deployableProperty == null) {
            logger.info("Deployed [{}] has embedded deployeds in property [{}] but source [{}] has no such property.", deployed.getId(), deployedPropDesc.getFqn(), deployable.getId());
            return;
        }
        @SuppressWarnings("unchecked")
        Collection<EmbeddedDeployable> embeddedDeployables = (Collection<EmbeddedDeployable>) deployableProperty.get(deployable);

        for (EmbeddedDeployable embeddedDeployable : embeddedDeployables) {
            final List<Type> embeddedDeployedTypes = calculator.findAllMostSpecificDeployedTypesForDeployableAndContainerTypes(embeddedDeployable.getType(), deployed.getType());
            Collection<Type> filtered = Collections2.filter(embeddedDeployedTypes, input -> input.instanceOf(deployedPropDesc.getReferencedType()));
            if (filtered.isEmpty()) {
                logger.info("Not found a matching EmbeddedDeployed type for [{}] and [{}]", embeddedDeployable.getType(), deployed.getType());
            } else if (filtered.size() > 1) {
                logger.error("Found more than 1 ({}) compatible EmbeddedDeployed type for [{}] and [{}]", filtered, embeddedDeployable.getType(), deployed.getType());
                throw new Checks.IncorrectArgumentException("Found more than 1 (%s) compatible EmbeddedDeployed type for [%s] and [%s]", filtered, embeddedDeployable.getType(), deployed.getType());
            } else {
                Type embeddedDeployedType = filtered.iterator().next();
                EmbeddedDeployed embeddedDeployed = embeddedDeployedType.getDescriptor().newInstance(IdGenerator.generateId(deployed, embeddedDeployable.getId()));
                //noinspection unchecked
                embeddedDeployed.setDeployable(embeddedDeployable);
                //noinspection unchecked
                embeddedDeployed.setContainer(deployed);

                if (MustachePlaceholderScanner.hasPlaceholders(embeddedDeployed.getName())) {
                    try {
                        dictionary.resolveDeployedName(embeddedDeployed);
                    } catch (DictionaryValueException e) {
                        throw new DeployedApplicationFactory.IncorrectDeployedException(e, "Couldn't generate name for target from [%s]", embeddedDeployable.getId());
                    }
                }
                embeddedDeployeds.add(embeddedDeployed);
                setProperties(embeddedDeployed, null, dictionary, missingPlaceholdersAggregator);
            }
        }

        deployedPropDesc.set(deployed, embeddedDeployeds);
    }

    private boolean isEmbeddedProperty(PropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAsContainment() &&
                (propertyDescriptor.getKind() == PropertyKind.SET_OF_CI || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI) &&
                propertyDescriptor.getReferencedType().isSubTypeOf(Type.valueOf(EmbeddedDeployed.class));
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployedPropertySetter.class);
}
