package com.xebialabs.deployit.jcr.grouping;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.task.archive.ArchivedTask;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newTreeSet;

public class GroupBy {

    private final List<Function<?>> functions;
    private final Map<String, Map<String, Object>> result = newHashMap();
    private final Set<String> propertyNames;
    private final Map<String, Field> fields;

    public GroupBy(Collection<String> properties, Function<?>... functions) {
        this.fields = getFields(properties);
        this.propertyNames = newTreeSet(properties);
        this.functions = functions != null ? Lists.<Function<?>> newArrayList(functions) : Lists.<Function<?>> newArrayList();
    }

    private Map<String, Field> getFields(Collection<String> properties) {
        Map<String, Field> fields = newHashMap();
        for (String property : properties) {
            try {
                Field field = ArchivedTask.class.getDeclaredField(property);
                field.setAccessible(true);
                fields.put(property, field);
            } catch (NoSuchFieldException e) {
                // Will be fetched from the metadata
            }
        }
        return fields;
    }

    public void process(ArchivedTask task) {
        String key = buildKey(task);
        Map<String, Object> newRow = this.result.get(key);
        if (newRow == null) {
            newRow = newHashMap();
            for (String property : propertyNames) {
                try {
                    newRow.put(property, getPropertyValue(task, property));
                } catch (IllegalAccessException e) {
                    // all are accessible
                }
            }
            this.result.put(key, newRow);
        }

        executeFunctions(task, key, newRow);
    }

    private Object getPropertyValue(ArchivedTask task, String property) throws IllegalAccessException {
        Field field = fields.get(property);
        if (field != null) {
            return field.get(task);
        } else {
            return task.getMetadata().get(property);
        }
    }

    private String buildKey(ArchivedTask task) {
        List<String> keyParts = newArrayList();
        for (String property : propertyNames) {
            try {
                keyParts.add(getPropertyValue(task, property).toString());
            } catch (Exception e) {
                // ok
            }
        }
        String key = Joiner.on("_").join(keyParts);
        return key;
    }

    private void executeFunctions(ArchivedTask task, String key, Map<String, Object> newRow) {
        for (Function<?> function : functions) {
            newRow.put(function.getName(), function.invoke(newRow.get(function.getName()), task));
        }
    }

    public Collection<Map<String, Object>> getResult() {
        return result.values();
    }
}
