package com.xebialabs.deployit.core.upgrade

import java.io.{IOException, InputStream}
import java.security.MessageDigest
import java.util
import javax.jcr.{Node, RepositoryException, Session}

import com.google.common.base.Strings.isNullOrEmpty
import com.google.common.io.ByteSource
import com.xebialabs.deployit.io.Exploder.calculateCheckSum
import com.xebialabs.deployit.jcr.JcrConstants
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry.getSubtypes
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact._
import com.xebialabs.deployit.repository.JcrPathHelper
import com.xebialabs.deployit.server.api.repository.RawRepository
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, UpgradeException, Version}
import com.xebialabs.deployit.upgrade.RawRepositoryImpl
import com.xebialabs.deployit.util.JavaCryptoUtils
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._
import scala.util.{Failure, Success, Try}

class Deployit500FolderChecksumUpgrader extends Upgrade with Logging {
  val DUMMY_CHECKSUM = "deadbeefbaadf00dd15ea5edcafebabebaddcafe"

  def doUpgrade(repository: RawRepository): Boolean = Try {
    logger.info("*** Running XL Deploy 5.0.0 Upgrade: Update Folder Checksums ***")
    upgradeAllFolderNodes(repository)
    logger.info("*** Done XL Deploy 5.0.0 upgrade -- Update Folder Checksums ***")
  } match {
    case Failure(e) => throw new UpgradeException(s"Failed to run ${this.getClass.getSimpleName}.", e.asInstanceOf[Exception])
    case Success(_) => true
  }

  private def upgradeAllFolderNodes(implicit repository: RawRepository) = {
    val folderType = Type.valueOf(classOf[FolderArtifact])
    val folderTypes = getSubtypes(folderType).toList :+ folderType
    logger.info(s"Upgrading artifacts of these types: $folderTypes")
    folderTypes.foreach { artifactType =>
      logger.info(s"Searching all nodes of type $artifactType")
      val nodes = repository.findNodesByType(artifactType)
      logger.info(s"Found ${nodes.size()} candidate node(s) of type $artifactType")
      if (nodes.size() > 0) logger.info(s"Calculating checksums, please wait...")
      batch(nodes) { node => updateNode(node) }
    }
  }

  private def updateNode(node: Node) = {
    if (!node.hasProperty(CHECKSUM_PROPERTY_NAME)) {
      updateChecksum(node, "[unset]")
    } else {
      val storedChecksum = node.getProperty(CHECKSUM_PROPERTY_NAME).getString
      if (isNullOrEmpty(storedChecksum)) {
        updateChecksum(node, "[empty]")
      } else {
        val ciId = JcrPathHelper.getIdFromAbsolutePath(node.getPath)
        calculateChecksum(node, digestPaths = false) match {
          case Success(calculatedChecksum) if storedChecksum == calculatedChecksum =>
            updateChecksum(node, storedChecksum)
          case Success(calculatedChecksum) =>
            logger.debug(s"Checksum of $ciId already updated or set manually - no update needed")
          case Failure(exc) =>
            logger.warn(s"Could not calculate old-style checksum for $ciId to compare to stored value; assuming update needed!")
            logger.debug(s"This is the error while calculating old-style checksum for $ciId: ", exc)
            updateChecksum(node, storedChecksum)
        }
      }
    }
  }


  private def updateChecksum(node: Node, storedChecksum: String) = {
    val ciId = JcrPathHelper.getIdFromAbsolutePath(node.getPath)
    val newChecksum = calculateChecksum(node, digestPaths = true) match {
      case Success(checksum) => checksum
      case Failure(exc) =>
        logger.error(s"Could not calculate new-style checksum for $ciId; using DUMMY value [$DUMMY_CHECKSUM]!")
        logger.debug(s"This is the error while calculating new-style checksum for $ciId: ", exc)
        DUMMY_CHECKSUM
    }
    logger.info(s"Updating $CHECKSUM_PROPERTY_NAME property of $ciId from $storedChecksum to $newChecksum")
    node.setProperty(CHECKSUM_PROPERTY_NAME, newChecksum)
  }

  private def calculateChecksum(node: Node, digestPaths: Boolean) = Try {
    val sha1: MessageDigest = JavaCryptoUtils.getSha1
    calculateCheckSum(getDataInputSupplier(node), sha1, digestPaths)
    JavaCryptoUtils.digest(sha1)
  }

  private def getDataInputSupplier(node: Node) = new ByteSource {
    def openStream: InputStream = {
      try {
        node.getProperty(JcrConstants.DATA_PROPERTY_NAME).getBinary.getStream
      } catch {
        case e: RepositoryException => throw new IOException(e)
      }
    }
  }

  def batch(nodes: util.List[Node], batchSize: Int = 500)(processNode: Node => Unit)(implicit repository: RawRepository) {
    val session: Session = repository.asInstanceOf[RawRepositoryImpl].getSession
    for (i <- 0 to nodes.size() - 1) {
      processNode(nodes(i))
      if (i > 0 && i % batchSize == 0) {
        session.save()
      }
    }
    session.save()
  }

  override def shouldBeApplied(currentVersion: Version): Boolean = {
    // upgrade has already been executed in the 3.9.6 / 4.0.3 / 4.5.3 versions which contain the same backported upgrade
    (currentVersion.getComponent == upgradeVersion().getComponent) && currentVersion.compareTo(upgradeVersion()) < 0 &&
      !(currentVersion.getVersion.startsWith("3.9.") && currentVersion.getMicro >= 6) &&
      !(currentVersion.getVersion.startsWith("4.0.") && currentVersion.getMicro >= 3) &&
      !(currentVersion.getVersion.startsWith("4.5.") && currentVersion.getMicro >= 3)
  }

  def upgradeVersion(): Version = Version.valueOf("deployit", "5.0.0")
}
