package com.xebialabs.deployit.task.archive

import javax.jcr.Node

import com.xebialabs.deployit.engine.api.execution.{StepState, CompositeBlockState, BlockState, TaskWithBlock}
import com.xebialabs.deployit.engine.tasker.{StepBlock, Phase, PhaseContainer}
import com.xebialabs.deployit.jcr.JcrConstants._
import com.xebialabs.deployit.jcr.JcrUtils._
import com.xebialabs.deployit.task.TaskMetadata._
import scala.collection.convert.wrapAll._

import com.xebialabs.deployit.task.archive.BlockType._

trait TaskArchiver {

  import NodeNames._

  def task: TaskWithBlock

  def save(rootNode: Node)

  protected def createTaskNode(rootNode: Node): Node = {
    val taskType = getOrCreateChild(rootNode, getMetadata(task, TASK_TYPE))
    val node = taskType.addNode(task.getId)
    node.addMixin(TASK_NODETYPE_NAME)

    node
  }

  protected def fillTaskNode(taskNode: Node, fillBlocks: Boolean = true) {
    val block = task.getBlock

    taskNode.setProperty(TASK_ID_PROPERTY_NAME, task.getId)
    taskNode.setProperty(STATE, task.getState.name())
    taskNode.setProperty(START_DATE, task.getStartDate.toGregorianCalendar)
    taskNode.setProperty(COMPLETION_DATE, task.getCompletionDate.toGregorianCalendar)
    taskNode.setProperty(OWNING_USER, task.getOwner)
    taskNode.setProperty(FAILURE_COUNT, task.getFailureCount)
    taskNode.setProperty(DEPLOYMENT_TYPE, getMetadata(task, TASK_TYPE))

    val scheduledDate = task.getScheduledDate
    if (scheduledDate != null) {
      taskNode.setProperty(SCHEDULED_DATE, scheduledDate.toGregorianCalendar)
    }

    if (fillBlocks) {
      fillRootBlock(taskNode, block)
    }
  }

  private def fillRootBlock(taskNode: Node, block: BlockState) {
    if (block != null) {
      fillBlockNode(taskNode, block)
    }
  }

  private def fillBlockNode(taskNode: Node, block: BlockState) {
    val blockNode = taskNode.addNode(block.getId)
    blockNode.setProperty(BLOCK_DESCRIPTION, block.getDescription)
    blockNode.setProperty(BLOCK_STATE, block.getState.name())

    block match {
      case container: PhaseContainer =>
        fillPhaseContainer(blockNode, container)

      case phase: Phase =>
        fillPhase(blockNode, phase)

      case stepBlock: StepBlock =>
        fillStepBlock(blockNode, stepBlock)

      case compositeBlock: CompositeBlockState =>
        fillCompositeBlock(blockNode, compositeBlock)

    }
  }

  private def fillPhaseContainer(blockNode: Node, container: PhaseContainer) {
    blockNode.setProperty(BLOCK_TYPE, PHASE_BLOCK_CONTAINER.name())
    for (p <- container.getPhases) {
      fillBlockNode(blockNode, p)
    }
  }

  private def fillPhase(blockNode: Node, phase: Phase) {
    blockNode.setProperty(BLOCK_TYPE, PHASE_BLOCK.name())
    fillBlockNode(blockNode, phase.block)
  }

  private def fillCompositeBlock(blockNode: Node, block: CompositeBlockState) {
    blockNode.setProperty(BLOCK_TYPE, COMPOSITE_BLOCK.name())
    blockNode.setProperty(IS_PARALLEL, block.isParallel)
    for (b <- block.getBlocks) {
      fillBlockNode(blockNode, b)
    }
  }

  private def fillStepBlock(blockNode: Node, block: StepBlock) {
    blockNode.setProperty(BLOCK_TYPE, STEP_BLOCK.name())
    blockNode.setProperty(CURRENT_STEP_NR_BLOCK, 0)
    fillStepNode(blockNode, block.getSteps())
  }

  private def fillStepNode(taskNode: Node, steps: java.util.List[StepState]) {
    var nr = 1
    for (step <- steps) {
      val stepNode = taskNode.addNode(STEP_NODE_NAME_PREFIX + nr)
      nr += 1
      stepNode.setProperty(DESCRIPTION, step.getDescription)
      stepNode.setProperty(STATE, step.getState.name())
      stepNode.setProperty(START_DATE, if (step.getStartDate == null) null else step.getStartDate.toGregorianCalendar)
      stepNode.setProperty(COMPLETION_DATE, if (step.getCompletionDate == null) null else step.getCompletionDate.toGregorianCalendar)
      stepNode.setProperty(FAILURE_COUNT, step.getFailureCount)
      stepNode.setProperty(LOG, step.getLog)
    }
  }

}

object TaskArchiver {

  def apply(task: TaskWithBlock): TaskArchiver = task match {
    case CloudTask(cloudArchiver) =>
      cloudArchiver

    case DeploymentTask(deploymentArchiver) =>
      deploymentArchiver

    case ControlTask(controlTask) =>
      controlTask

    case _ =>
      new DefaultTaskArchiver(task)
  }
}