package com.xebialabs.deployit.task.archive

import java.util
import javax.jcr.Node
import javax.jcr.nodetype.NodeType

import com.google.common.base.Preconditions._
import com.google.common.base.Strings
import com.xebialabs.deployit.engine.api.execution.{TaskPackageDependency, TaskWithBlock}
import com.xebialabs.deployit.jcr.JcrConstants._
import com.xebialabs.deployit.jcr.JcrUtils._
import com.xebialabs.deployit.task.TaskMetadata._
import com.xebialabs.deployit.task.TaskType.{INITIAL, ROLLBACK, UNDEPLOY, UPGRADE}
import com.xebialabs.deployit.task.archive.NodeNames.{DEPLOYMENT_TYPE, ENVIRONMENT_ID, ENVIRONMENT_NAME}
import com.xebialabs.deployit.task.{TaskMetadata, TaskType}

import scala.collection.convert.wrapAll._

class DeploymentTaskArchiver(val task: TaskWithBlock) extends TaskArchiver {

  override def save(rootNode: Node) {
    val mainTaskNode = buildTaskNode(rootNode, getMetadata(task, APPLICATION), getMetadata(task, VERSION), persistSteps = true)
    buildTaskRefs(rootNode, mainTaskNode, task.getId)
    mainTaskNode.setProperty(NodeNames.PACKAGE_DEPENDENCY_NAMES, task.getPackageDependencies.map(_.getApplication).toArray)
    mainTaskNode.setProperty(NodeNames.PACKAGE_DEPENDENCY_NAMES_AND_VERSIONS, task.getPackageDependencies.map { dependency =>
      s"${dependency.getApplication}/${dependency.getVersion}"
    }.toArray)
  }

  private def buildTaskNode(rootNode: Node, application: String, version: String, persistSteps: Boolean) = {
    val environment = getMetadata(task, ENVIRONMENT)
    val environmentId = EnvironmentId.encode(getMetadata(task, TaskMetadata.ENVIRONMENT_ID))

    //"Environments/myfolder/env/app/1.0/"
    val envNode = getOrCreateChild(rootNode, environmentId)
    val appNode = getOrCreateChild(envNode, application)
    val verNode = getOrCreateChild(appNode, version)
    val taskNode = verNode.addNode(task.getId)

    taskNode.addMixin(TASK_NODETYPE_NAME)
    taskNode.addMixin(NodeType.MIX_REFERENCEABLE)
    fillTaskNode(taskNode, fillBlocks = persistSteps)
    fillInAppInfo(taskNode, application, version, environment, environmentId)
    taskNode
  }

  private def buildTaskRefs(rootNode: Node, mainTask: Node, mainTaskId: String) {
    task.getPackageDependencies.foreach { dependency =>
      val refNode = buildTaskNode(rootNode, dependency.getApplication, dependency.getVersion, persistSteps = false)
      refNode.setProperty(NodeNames.PARENT_TASK, mainTask)
      refNode.setProperty(NodeNames.PARENT_TASK_ID, mainTaskId)
    }
  }

  private def fillInAppInfo(node: Node, application: String, version: String, environment: String, environmentId: String) {
    node.setProperty(APPLICATION, application)
    node.setProperty(VERSION, version)
    node.setProperty(ENVIRONMENT_NAME, environment)
    node.setProperty(ENVIRONMENT_ID, environmentId)
  }
}

class DeploymentTaskReader(val taskNode: Node) extends TaskReader {
  override def readWithoutStep(): ArchivedTask = {
    val task = taskWithoutSteps

    putMetadata(task, ENVIRONMENT, getProperty(taskNode, ENVIRONMENT_NAME))
    if (taskNode.hasProperty(ENVIRONMENT_ID)) {
      putMetadata(task, ENVIRONMENT_ID, EnvironmentId.decode(getProperty(taskNode, ENVIRONMENT_ID)))
    } else {
      // Support old nodes that do not have the environment_id set.
      putMetadata(task, ENVIRONMENT_ID, EnvironmentId.decode(getProperty(taskNode, ENVIRONMENT_NAME)))
    }
    putMetadata(task, APPLICATION, getProperty(taskNode, APPLICATION))
    putMetadata(task, VERSION, getProperty(taskNode, VERSION))
    task.getPackageDependencies.addAll(readTaskRefs(taskNode))

    task
  }

  private def readTaskRefs(taskNode: Node) = {
    if (taskNode.hasProperty(NodeNames.PACKAGE_DEPENDENCY_NAMES_AND_VERSIONS)) {
      taskNode.getProperty(NodeNames.PACKAGE_DEPENDENCY_NAMES_AND_VERSIONS).getValues.toList.map { nameAndVersion =>
        val split = nameAndVersion.getString.split("/")
        new TaskPackageDependency(split(0), split(1))
      }
    } else Nil
  }
}

object DeploymentTask {

  def unapply(task: TaskWithBlock): Option[TaskArchiver] = {
    val taskType = TaskType.valueOf(getMetadata(task, TASK_TYPE))
    if (util.EnumSet.of(ROLLBACK, INITIAL, UPGRADE, UNDEPLOY).contains(taskType)) {

      checkState(!Strings.nullToEmpty(getMetadata(task, APPLICATION)).isEmpty, "applicationName in deployment task must be set", "")
      checkState(!Strings.nullToEmpty(getMetadata(task, VERSION)).isEmpty, "applicationVersion in deployment task must be set", "")
      checkState(!Strings.nullToEmpty(getMetadata(task, ENVIRONMENT)).isEmpty, "environment in deployment task must be set", "")
      checkState(!Strings.nullToEmpty(getMetadata(task, TaskMetadata.ENVIRONMENT_ID)).isEmpty, "environment_id in deployment task must be set", "")

      Some(new DeploymentTaskArchiver(task))
    } else {
      None
    }
  }

  def unapply(taskNode: Node): Option[TaskReader] = {
    val taskType = TaskType.valueOf(getProperty(taskNode, DEPLOYMENT_TYPE))

    if (util.EnumSet.of(ROLLBACK, INITIAL, UPGRADE, UNDEPLOY).contains(taskType)) {
      Some(new DeploymentTaskReader(taskNode))
    } else None
  }

}